package ru.yandex.wmconsole.service;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;

import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.service.AbstractDbService;

/**
 * @author Andrey Mima (amima@yandex-team.ru)
 */
public class PublicReindexUrlService extends AbstractDbService {
    private static final Logger log = LoggerFactory.getLogger(PublicReindexUrlService.class);

    private static final String FIELD_REQUESTS = "requests_count";
    private static final String FIELD_LAST_REQUEST = "last_request";

    private int maxReindexPerDay;

    private static final String SELECT_REINDEX_LIMITS_QUERY =
            "SELECT " +
                    "   requests_count AS " + FIELD_REQUESTS + ", " +
                    "   last_request AS " + FIELD_LAST_REQUEST + " " +
                    "FROM " +
                    "   tbl_host_reindex_limits " +
                    "WHERE " +
                    "   host_id = ? ";

    private static final String UPDATE_REINDEX_REQUESTS_RESET_QUERY =
            "UPDATE " +
                    "   tbl_host_reindex_limits " +
                    "SET " +
                    "   requests_count = 1, " +
                    "   last_request = NOW() " +
                    "WHERE " +
                    "   host_id = ? ";

    private static final String UPDATE_REINDEX_REQUESTS_INCREMENT_QUERY =
            "UPDATE " +
                    "   tbl_host_reindex_limits " +
                    "SET " +
                    "   requests_count = requests_count + 1, " +
                    "   last_request = NOW() " +
                    "WHERE " +
                    "   host_id = ? ";

    private static final String INSERT_REINDEX_CREATE_LIMITS_QUERY =
            "INSERT INTO " +
                    "   tbl_host_reindex_limits " +
                    "   (host_id, requests_count, last_request) " +
                    "VALUES " +
                    " (?, 1, NOW()) ";

    private static class ReindexLimitsInfo {
        private final int requests;
        private final Date lastRequest;

        private ReindexLimitsInfo(int requests, Date lastRequest) {
            this.requests = requests;
            this.lastRequest = lastRequest;
        }

        public int getRequests() {
            return requests;
        }

        public Date getLastRequest() {
            return lastRequest;
        }
    }

    private static final ParameterizedRowMapper<ReindexLimitsInfo> reindexLimitsMapper =
            new ParameterizedRowMapper<ReindexLimitsInfo>() {
        @Override
        public ReindexLimitsInfo mapRow(ResultSet resultSet, int rowNum) throws SQLException {
            Integer requests = resultSet.getInt(FIELD_REQUESTS);
            Date lastRequest = resultSet.getDate(FIELD_LAST_REQUEST);

            return new ReindexLimitsInfo(requests, lastRequest);
        }
    };

    public boolean isLimitExceededAndIncrement(Long hostId) throws InternalException {
        List<ReindexLimitsInfo> reindexLimitsQueryResult = getJdbcTemplate(WMCPartition.nullPartition()).query(
                SELECT_REINDEX_LIMITS_QUERY, reindexLimitsMapper, hostId);
        if (reindexLimitsQueryResult.size() == 0) {
            getJdbcTemplate(WMCPartition.nullPartition()).update(INSERT_REINDEX_CREATE_LIMITS_QUERY, hostId);
            return false;
        }

        ReindexLimitsInfo hostLimits = reindexLimitsQueryResult.get(0);
        if (hostLimits.getRequests() >= maxReindexPerDay) {
            Calendar now = Calendar.getInstance();
            Calendar hostLastAccess = Calendar.getInstance();
            hostLastAccess.setTime(hostLimits.getLastRequest());
            hostLastAccess.add(Calendar.DAY_OF_MONTH, 1);
            if (hostLastAccess.before(now)) {
                getJdbcTemplate(WMCPartition.nullPartition()).update(UPDATE_REINDEX_REQUESTS_RESET_QUERY, hostId);
                return false;
            } else {
                return true;
            }
        } else {
            getJdbcTemplate(WMCPartition.nullPartition()).update(UPDATE_REINDEX_REQUESTS_INCREMENT_QUERY, hostId);
            return false;
        }
    }

    public int getRequestsCount(Long hostId) throws InternalException {
        List<ReindexLimitsInfo> reindexLimitsQueryResult = getJdbcTemplate(WMCPartition.nullPartition()).query(
                SELECT_REINDEX_LIMITS_QUERY, reindexLimitsMapper, hostId);
        if (reindexLimitsQueryResult.size() == 0) {
            return 0;
        }

        ReindexLimitsInfo hostLimits = reindexLimitsQueryResult.get(0);

        if (hostLimits.getRequests() >= maxReindexPerDay) {
            Calendar now = Calendar.getInstance();
            Calendar hostLastAccess = Calendar.getInstance();
            hostLastAccess.setTime(hostLimits.getLastRequest());
            hostLastAccess.add(Calendar.DAY_OF_MONTH, 1);
            if (hostLastAccess.before(now)) {
                return 0;
            }
        }

        return hostLimits.getRequests();
    }

    @Required
    public void setMaxReindexPerDay(int maxReindexPerDay) {
        this.maxReindexPerDay = maxReindexPerDay;
    }
}
