package ru.yandex.wmconsole.service;

import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.common.framework.user.UserInfo;
import ru.yandex.common.util.concurrent.CommonThreadFactory;
import ru.yandex.common.util.db.LongRowMapper;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.service.BlackboxUserService;
import ru.yandex.wmtools.common.util.SqlUtil;

/**
 * @author avhaliullin
 */
public class RemovedUsersService extends AbstractDbService implements BlackboxUserService.UserInfoRequestHandler {
    private static final Logger log = LoggerFactory.getLogger(RemovedUsersService.class);

    private static final String INSERT_MARK_USERS_REMOVED_QUERY =
            "INSERT IGNORE INTO " +
                    "      tbl_removed_users (user_id, last_check) " +
                    "   VALUES " +
                    "       %1$s ";

    private static final String SELECT_USERS_FROM_DB_QUERY =
            "SELECT " +
                    "       user_id " +
                    "   FROM " +
                    "       tbl_users " +
                    "   WHERE " +
                    "       user_id IN (%1$s)";

    private final ConcurrentSkipListSet<Long> usersToRemove = new ConcurrentSkipListSet<Long>();
    private ScheduledExecutorService scheduler;

    public void init() {
        CommonThreadFactory threadFactory = new CommonThreadFactory(true, RemovedUsersService.class.getSimpleName() + "-");
        scheduler = Executors.newSingleThreadScheduledExecutor(threadFactory);

        scheduler.scheduleWithFixedDelay(new Runnable() {
            @Override
            public void run() {
                if (usersToRemove.isEmpty()) {
                    return;
                }
                try {
                    Set<Long> toRemove = usersToRemove.clone();
                    String selectQuery = String.format(SELECT_USERS_FROM_DB_QUERY, SqlUtil.getCommaSeparatedList(toRemove));
                    List<Long> toRemoveFromDb = getJdbcTemplate(WMCPartition.nullPartition()).query(selectQuery,
                            new LongRowMapper());

                    if (!toRemoveFromDb.isEmpty()) {
                        String insertQuery = String.format(INSERT_MARK_USERS_REMOVED_QUERY,
                                SqlUtil.getCommaSeparatedList(toRemoveFromDb, new SqlUtil.ListParameterizer<Long>() {
                                    @Override
                                    public String getParameter(int i, Long obj) {
                                        return i == 0 ? String.valueOf(obj) : "NOW()";
                                    }

                                    @Override
                                    public int getParamNumber() {
                                        return 2;
                                    }
                                }));
                        getJdbcTemplate(WMCPartition.nullPartition()).update(insertQuery);
                    }
                    usersToRemove.removeAll(toRemove);
                } catch (InternalException e) {
                    log.warn("Failed to insert removed users", e);
                }
            }
        }, 1, 1, TimeUnit.MINUTES);
    }

    public void removeUser(long userId) {
        usersToRemove.add(userId);
    }

    @Override
    public void handleRequest(Long userId, UserInfo response) {
        if (response == null) {
            removeUser(userId);
        }
    }
}
