package ru.yandex.wmconsole.service;

import java.net.MalformedURLException;
import java.net.URL;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.wmconsole.service.error.WMCUserProblem;
import ru.yandex.wmtools.common.Constants;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.error.UserProblem;
import ru.yandex.wmtools.common.service.IService;
import ru.yandex.wmtools.common.sita.DocumentFormatEnum;
import ru.yandex.wmtools.common.sita.RobotsTxtFormatEnum;
import ru.yandex.wmtools.common.sita.SitaService;
import ru.yandex.wmtools.common.sita.SitaUrlFetchRequest;
import ru.yandex.wmtools.common.sita.SitaUrlFetchRequestBuilder;
import ru.yandex.wmtools.common.sita.SitaUrlFetchResponse;
import ru.yandex.wmtools.common.sita.UserAgentEnum;
import ru.yandex.wmtools.common.util.http.YandexHttpStatus;

/**
 * This service works with robots.txt files.
 *
 * @author ailyin
 */
public class RobotsTxtService implements IService, Constants {
    private static final Logger log = LoggerFactory.getLogger(RobotsTxtService.class);

    private static final String ROBOTS_TXT_PATH = "/robots.txt";

    private SitaService sitaService;

    public String getRobotsTxtContent(URL host) throws UserException, InternalException {
        final String robotsTxtContent;
        URL robotsTxtUrl;
        try {
            robotsTxtUrl = getRobotsTxtUrl(host);
        } catch (MalformedURLException e) {
            log.debug("Unable to get robots.txt content", e);
            throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "Unable to create robots.txt url", e);
        }

        SitaUrlFetchRequest sitaUrlFetchRequest = new SitaUrlFetchRequestBuilder(robotsTxtUrl)
                .setCheckAllowedInRobotsTxt(true)
                .setDocumentFormat(DocumentFormatEnum.DF_NO_DOC)
                .setRobotsTxtFormat(RobotsTxtFormatEnum.RF_TEXT)
                .setUserAgent(UserAgentEnum.ROBOT)
                .createSitaUrlFetchRequest();
        SitaUrlFetchResponse sitaUrlFetchResponse = sitaService.request(sitaUrlFetchRequest);
        if (YandexHttpStatus.is3XX(sitaUrlFetchResponse.getSitaHttpStatus())) {
            throw new UserException(UserProblem.ROBOTS_TXT_REDIRECTED, "Request to robots.txt redirected");
        }

        if (sitaUrlFetchResponse.getSitaHttpStatus() != YandexHttpStatus.HTTP_200_OK) {
            log.error("Wrong status code: {}", sitaUrlFetchResponse.getSitaHttpStatus());
            throw new UserException(WMCUserProblem.ROBOTS_TXT_ACCESS_ERROR, "Failed to load robots.txt");
        }

        robotsTxtContent = sitaUrlFetchResponse.getRobotsTxtContent();
        if (StringUtils.isBlank(robotsTxtContent)) {
            throw new UserException(WMCUserProblem.EMPTY_ROBOTS_TXT, "File robots.txt does not contain rules.");
        }
        return robotsTxtContent;
    }

    private URL getRobotsTxtUrl(URL url) throws MalformedURLException {
        return new URL(url.getProtocol() + SCHEME_DELIMITER + url.getAuthority() + ROBOTS_TXT_PATH);
    }

    @Required
    public void setNewSitaService(SitaService sitaService) {
        this.sitaService = sitaService;
    }
}
