package ru.yandex.wmconsole.service;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.transaction.TransactionStatus;

import ru.yandex.wmconsole.data.ServiceEnum;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.ServiceOptionsInfo;
import ru.yandex.wmconsole.data.info.ShortServiceOptionsInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.util.ServiceTransactionCallbackWithoutResult;

/**
 * @author ailyin
 */
public class ServiceOptionsService extends AbstractDbService {
    private static final Logger log = LoggerFactory.getLogger(ServiceOptionsService.class);

    private static final String FIELD_SERVICE_ID = "service_id";
    private static final String FIELD_PUBLIC = "public";
    private static final String FIELD_ENABLED = "enabled";

    private static final String SELECT_SERVICE_OPTIONS_FOR_HOST_QUERY =
            "SELECT " +
                    "   s.id AS " + FIELD_SERVICE_ID + ", " +
                    "   s.public AS " + FIELD_PUBLIC + ", " +
                    "   hso.enabled AS " + FIELD_ENABLED + " " +
                    "FROM " +
                    "   tbl_dic_service s " +
                    "LEFT JOIN " +
                    "   tbl_host_service_options hso " +
                    "ON " +
                    "   s.id = hso.service_id " +
                    "AND " +
                    "   host_id = ? ";

    private static final String SELECT_ALL_SERVICES_QUERY =
            "SELECT " +
                    "   s.id AS " + FIELD_SERVICE_ID + ", " +
                    "   s.public AS " + FIELD_PUBLIC + " " +
                    "FROM " +
                    "   tbl_dic_service s";

    private static final String UPDATE_OPTIONS_FOR_HOST_QUERY =
            "REPLACE INTO " +
                    "   tbl_host_service_options " +
                    "   (host_id, service_id, enabled) " +
                    "VALUES " +
                    "   %s ";

    private static final ParameterizedRowMapper<ServiceOptionsInfo> serviceOptionsMapper =
            new ParameterizedRowMapper<ServiceOptionsInfo>() {
        @Override
        public ServiceOptionsInfo mapRow(ResultSet resultSet, int rowNum) throws SQLException {
            ServiceEnum service = ServiceEnum.R.fromValueOrNull(resultSet.getInt(FIELD_SERVICE_ID));
            boolean enabled = resultSet.getBoolean(FIELD_ENABLED);
            if (resultSet.wasNull()) {
                enabled = service.isEnabledByDefault();
            }
            boolean isPublic = resultSet.getBoolean(FIELD_PUBLIC);

            return new ServiceOptionsInfo(service, enabled, isPublic);
        }
    };

    private static final ParameterizedRowMapper<ServiceOptionsInfo> allServicesMapper =
            new ParameterizedRowMapper<ServiceOptionsInfo>() {
                @Override
                public ServiceOptionsInfo mapRow(ResultSet resultSet, int rowNum) throws SQLException {
                    ServiceEnum service = ServiceEnum.R.fromValueOrNull(resultSet.getInt(FIELD_SERVICE_ID));
                    boolean isPublic = resultSet.getBoolean(FIELD_PUBLIC);

                    return new ServiceOptionsInfo(service, false, isPublic);
                }
            };

    public List<ServiceOptionsInfo> getServicesOptions(BriefHostInfo hostInfo) throws InternalException {
        return getJdbcTemplate(WMCPartition.nullPartition()).query(SELECT_SERVICE_OPTIONS_FOR_HOST_QUERY,
                serviceOptionsMapper, hostInfo.getId());
    }

    public List<ServiceEnum> getEnabledServices(BriefHostInfo hostInfo) throws InternalException {
        List<ServiceOptionsInfo> allOptions = getServicesOptions(hostInfo);

        List<ServiceEnum> result = new ArrayList<ServiceEnum>();
        for (ServiceOptionsInfo info : allOptions) {
            if (info.isEnabled()) {
                result.add(info.getService());
            }
        }
        return result;
    }

    public void updateOptions(final BriefHostInfo hostInfo, final List<ShortServiceOptionsInfo> newOptions)
            throws InternalException, UserException {
        getServiceTransactionTemplate(WMCPartition.nullPartition()).executeInService(
                new ServiceTransactionCallbackWithoutResult() {
            @Override
            protected void doInTransactionWithoutResult(TransactionStatus transactionStatus)
                    throws UserException, InternalException {
                List<ServiceOptionsInfo> currentOptions = getServicesOptions(hostInfo);

                boolean optionsChanged = false;
                StringBuilder builder = new StringBuilder();
                for (ServiceOptionsInfo current : currentOptions) {
                    for (ShortServiceOptionsInfo newOption : newOptions) {
                        if (newOption.getService().equals(current.getService()) &&
                                (newOption.isEnabled() != current.isEnabled())) {
                            log.debug(newOption.getService() + " option has been turned " +
                                    (newOption.isEnabled() ? "ON" : "OFF") + " for host " + hostInfo.getName());

                            if (optionsChanged) {
                                builder.append(",");
                            }
                            optionsChanged = true;
                            builder.append("(").append(hostInfo.getId()).append(",");
                            builder.append(newOption.getService().getValue()).append(",");
                            builder.append(newOption.isEnabled()).append(")");
                        }
                    }
                }

                if (optionsChanged) {
                    getJdbcTemplate(WMCPartition.nullPartition()).update(
                            String.format(UPDATE_OPTIONS_FOR_HOST_QUERY, builder.toString()));
                }
            }
        });
    }

    /**
     * Выдает список всех сервисов из справочника. При выдаче enabled = false для всех сервисов.
     * @return список сервисов с настройками
     * @throws InternalException
     */
    public List<ServiceOptionsInfo> getAllServices() throws InternalException {
        return getJdbcTemplate(WMCPartition.nullPartition()).query(
                SELECT_ALL_SERVICES_QUERY, allServicesMapper);
    }
}
