package ru.yandex.wmconsole.service;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.NavigableMap;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;

import ru.yandex.common.util.collections.Pair;
import ru.yandex.wmconsole.data.SpiderRequestResultEnum;
import ru.yandex.wmconsole.data.info.HostDbHostInfo;
import ru.yandex.wmconsole.data.info.spider.CodesGroup;
import ru.yandex.wmconsole.data.info.spider.SpiderCodesGroupFactory;
import ru.yandex.wmconsole.data.info.spider.SpiderCodesGroupInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.util.ParameterizedMapRowMapper;
import ru.yandex.wmtools.common.util.SqlUtil;
import ru.yandex.wmtools.common.util.TimeFilter;

/**
 * @author avhaliullin
 */
public class SpiderInfoService extends AbstractDbService {
    private static final Logger log = LoggerFactory.getLogger(SpiderInfoService.class);

    private static final String FIELD_HTTP_CODE = "code";
    private static final String FIELD_HOST_ID = "host_id";
    private static final String FIELD_TIME = "time_stamp";
    private static final String FIELD_COUNT = "count";

    private static final String FIELD_MAX_TIMESTAMP = "max";
    private static final String FIELD_MIN_TIMESTAMP = "min";

    private static final String SELECT_LIST_CODES_FOR_HOST_QUERY =
            "SELECT " +
                    "       DISTINCT hs.code AS " + FIELD_HTTP_CODE + " " +
                    "   FROM " +
                    "       tbl_hostmon_stats hs " +
                    "   WHERE " +
                    "       hs.host_id = ? " +
                    "       %1$s " +
                    "       %2$s " +
                    "   ORDER BY " +
                    "       hs.code ASC ";


    private static final String SELECT_DATE_RANGE_FOR_HOST_QUERY =
            "SELECT " +
                    "       MAX(time_stamp) AS " + FIELD_MAX_TIMESTAMP + ", " +
                    "       IF(MIN(time_stamp) < MAX(time_stamp) - INTERVAL 2 WEEK, MAX(time_stamp) - INTERVAL 2 WEEK, MIN(time_stamp)) AS " + FIELD_MIN_TIMESTAMP + " " +
                    "   FROM " +
                    "       tbl_hostmon_stats hs " +
                    "   WHERE " +
                    "       host_id = ?  ";

    private static final String SELECT_PLOT_DATA_QUERY =
            "SELECT " +
                    "       SUM(hs.count) AS " + FIELD_COUNT + ", " +
                    "       hs.time_stamp AS " + FIELD_TIME + " " +
                    "   FROM " +
                    "       tbl_hostmon_stats hs " +
                    "   WHERE " +
                    "       host_id = ? " +
                    "       %1$s " +
                    "       %2$s " +
                    "   GROUP BY " +
                    "       hs.time_stamp ";

    private List<SpiderCodesGroupFactory> groupFactories;

    private static final ParameterizedRowMapper<Pair<Date, Date>> dateRangeMapper = new ParameterizedRowMapper<Pair<Date, Date>>() {
        @Override
        public Pair<Date, Date> mapRow(ResultSet rs, int rowNum) throws SQLException {
            return Pair.<Date, Date>of(rs.getTimestamp(FIELD_MIN_TIMESTAMP), rs.getTimestamp(FIELD_MAX_TIMESTAMP));
        }
    };

    private Pair<Date, Date> getRange(HostDbHostInfo hostInfo) throws InternalException {
        return getJdbcTemplate(new WMCPartition(hostInfo, null))
                .queryForObject(SELECT_DATE_RANGE_FOR_HOST_QUERY, dateRangeMapper, hostInfo.getHostDbHostId());
    }

    private NavigableMap<Date, Long> getData(HostDbHostInfo hostInfo, String codesFilter, Pair<Date, Date> range) throws InternalException {
        String timeFilter = getTimeFilter(range);
        NavigableMap<Date, Long> map = getJdbcTemplate(new WMCPartition(hostInfo, null))
                .queryForNavigableMap(
                        String.format(SELECT_PLOT_DATA_QUERY,
                                timeFilter,
                                " AND (" + codesFilter + ")"),
                        httpCodesCountPlotMapper,
                        hostInfo.getHostDbHostId());
        if (!map.isEmpty()) {
            if (!map.containsKey(range.first)) {
                map.put(range.first, 0L);
            }
            if (!map.containsKey(range.second)) {
                map.put(range.second, 0L);
            }
        }
        return map;
    }

    public List<SpiderCodesGroupInfo> getCodesGroupInfos(List<String> requestedSubgroups) throws InternalException {
        Set<String> requestedSubgroupsSet = null;
        if (requestedSubgroups != null) {
            requestedSubgroupsSet = new HashSet<String>();
            for (String s : requestedSubgroups) {
                requestedSubgroupsSet.add(s.toLowerCase());
            }
        }
        List<SpiderCodesGroupInfo> groups = new ArrayList<SpiderCodesGroupInfo>();
        for (SpiderCodesGroupFactory factory : groupFactories) {
            SpiderCodesGroupInfo group = factory.createGroup(requestedSubgroupsSet);
            if (group != null) {
                groups.add(group);
            }
        }
        return groups;
    }

    public Pair<List<String>, List<NavigableMap<Date, Long>>> getHttpCodesGroupsPlotData(HostDbHostInfo hostDbHostInfo,
                                                                                         List<? extends CodesGroup> groups) throws InternalException {
        List<Pair<String, NavigableMap<Date, Long>>> tmpPlot = new ArrayList<Pair<String, NavigableMap<Date, Long>>>();
        Pair<Date, Date> range = getRange(hostDbHostInfo);

        for (CodesGroup group : groups) {
            tmpPlot.add(new Pair<String, NavigableMap<Date, Long>>(
                    group.getName(),
                    getData(hostDbHostInfo, group.getSqlFilter(), range)));
        }

        Collections.sort(tmpPlot, new Comparator<Pair<String, NavigableMap<Date, Long>>>() {
            @Override
            public int compare(Pair<String, NavigableMap<Date, Long>> o1, Pair<String, NavigableMap<Date, Long>> o2) {
                boolean i1 = o1.getSecond().isEmpty();
                boolean i2 = o2.getSecond().isEmpty();
                return i1 == i2 ? 0 : (i2 ? -1 : 1);
            }
        });
        Pair<List<String>, List<NavigableMap<Date, Long>>> res = new Pair<List<String>, List<NavigableMap<Date, Long>>>(
                new ArrayList<String>(), new ArrayList<NavigableMap<Date, Long>>());
        for (Pair<String, NavigableMap<Date, Long>> pair : tmpPlot) {
            res.getFirst().add(pair.getFirst());
            res.getSecond().add(pair.getSecond());
        }
        return res;
    }

    private static final ParameterizedRowMapper<Integer> httpCodeMapper = new ParameterizedRowMapper<Integer>() {
        @Override
        public Integer mapRow(ResultSet resultSet, int i) throws SQLException {
            return resultSet.getInt(FIELD_HTTP_CODE);
        }
    };

    private static final ParameterizedMapRowMapper<Date, Long> httpCodesCountPlotMapper = new ParameterizedMapRowMapper<Date, Long>() {
        @Override
        public Pair<Date, Long> mapRow(ResultSet resultSet, int i) throws SQLException {
            return new Pair<Date, Long>(resultSet.getTimestamp(FIELD_TIME), resultSet.getLong(FIELD_COUNT));
        }
    };


    private String getTimeFilter(Pair<Date, Date> range) throws InternalException {
        return SqlUtil.getTimeFilterPart(TimeFilter.create(range), "hs.time_stamp", true, true);
    }

    private List<Integer> getCodesForHost(HostDbHostInfo hostDbHostInfo, String filter) throws InternalException {
        return getJdbcTemplate(new WMCPartition(hostDbHostInfo, null))
                .query(
                        String.format(SELECT_LIST_CODES_FOR_HOST_QUERY,
                                getTimeFilter(getRange(hostDbHostInfo)),
                                filter),
                        httpCodeMapper,
                        hostDbHostInfo.getHostDbHostId());
    }

    public NavigableMap<Date, Long> getSpiderRequestResultsPlotData(HostDbHostInfo hostDbHostInfo, SpiderRequestResultEnum type) throws InternalException {
        return getData(hostDbHostInfo, "hs.code IN (" + type.getFilter() + ")", getRange(hostDbHostInfo));
    }


    public List<SpiderRequestResultEnum> getSpiderRequestResults(HostDbHostInfo hostDbHostInfo) throws InternalException {
        List<Integer> codes = getCodesForHost(hostDbHostInfo, " AND hs.code IN (3, 4, 5, 1010)");
        Set<SpiderRequestResultEnum> res = new HashSet<SpiderRequestResultEnum>();
        for (Integer code : codes) {
            SpiderRequestResultEnum type = SpiderRequestResultEnum.getByCode(code);
            if (type != null) {
                res.add(type);
            }
        }
        return new ArrayList<SpiderRequestResultEnum>(res);
    }

    @Required
    public void setGroupFactories(List<SpiderCodesGroupFactory> groupFactories) {
        this.groupFactories = groupFactories;
    }
}
