package ru.yandex.wmconsole.service;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Map;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import ru.yandex.wmtools.common.Constants;
import ru.yandex.wmtools.common.service.IService;
import ru.yandex.wmtools.common.util.HttpConnector;
import ru.yandex.wmtools.common.util.HttpResponse;

/**
 * @author yakushev
 */
public class TcyProviderService implements IService, Constants {
    private static final Logger log = LoggerFactory.getLogger(TcyProviderService.class);

    private Map<String,Integer> fixedTicValues;

    private String yandexBarRequestUrl;

    /**
     * @param hostname hostname
     * @return TCY value; null, if failed to retrieve it
     */
    public Integer getTcy(URL hostname) {
        Integer val = fixedTicValues.get(hostname.getHost().toLowerCase());
        if (val != null) {
            return val;
        }

        /*
         * There are no schemes in the tcy database so we have to make a request with 'http://' scheme
         */
        String requestUrl = yandexBarRequestUrl + HTTP_PREFIX + hostname.getAuthority();
        try {
            HttpResponse httpResponse = new HttpConnector.RequestBuilder(new URL(requestUrl)).execute();
            return extractTcy(httpResponse.getContent());
        } catch (IOException e) {
            log.error("Failed to connect to " + requestUrl, e);
            return null;
        } catch (SAXException e) {
            log.error("Failed to parse Yandex.Bar response for host " + hostname, e);
            return null;
        } catch (ParserConfigurationException e) {
            log.error("Failed to parse Yandex.Bar response for host " + hostname, e);
            return null;
        }
    }

    private Integer extractTcy(InputStream stream) throws IOException, SAXException, ParserConfigurationException {
        SAXParserFactory saxParserFactory = SAXParserFactory.newInstance();
        saxParserFactory.setNamespaceAware(true);
        SAXParser saxParser = saxParserFactory.newSAXParser();
        TcyXmlHandler handler = new TcyXmlHandler();
        saxParser.parse(new InputSource(stream), handler);
        return handler.getTcy();
    }

    private static class TcyXmlHandler extends DefaultHandler {
        private static final String TCY_TAG = "tcy";
        private static final String TCY_VALUE_ATTR = "value";

        private Integer tcy;

        @Override
        public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {
            if (!TCY_TAG.equalsIgnoreCase(localName)) {
                return;
            }

            String valueStr = attributes.getValue("", TCY_VALUE_ATTR);
            if (valueStr == null) {
                log.error("Unable to extract tcy value. " +
                        "Tag " + TCY_TAG + " hasn't got attribute with name " + TCY_VALUE_ATTR);
                return;
            }

            if (valueStr.isEmpty()) {
                tcy = 0;
                return;
            }

            try {
                tcy = Integer.valueOf(valueStr);
                if (tcy < 0) {
                    tcy = 0;
                }
            } catch (NumberFormatException e) {
                log.error("Unable to extract tcy value." +
                        "Attribute " + TCY_VALUE_ATTR + " has non digital value.", e);
            }
        }

        public Integer getTcy() {
            return tcy;
        }
    }

    public void addFixedTicValue(final String hostName, final int tic) {
        if (hostName == null) {
            log.error("addFixedTicValue hostName is null");
            return;
        }

        log.debug("adding fixed TIC " + tic + " for " + hostName.toLowerCase());
        fixedTicValues.put(hostName.toLowerCase(), tic);
    }

    public void removeFixedTicValue(final String hostName) {
        if (hostName == null) {
            log.error("removeFixedTicValue hostName is null");
            return;
        }

        log.debug("removing fixed TIC for " + hostName.toLowerCase());
        fixedTicValues.remove(hostName.toLowerCase());
    }

    @Required
    public void setFixedTicValues(Map<String, Integer> fixedTicValues) {
        this.fixedTicValues = fixedTicValues;
    }

    @Required
    public void setYandexBarRequestUrl(String yandexBarRequestUrl) {
        this.yandexBarRequestUrl = yandexBarRequestUrl;
    }
}
