package ru.yandex.wmconsole.service;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Required;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;

import ru.yandex.webmaster.common.urltree.YandexSearchShard;
import ru.yandex.wmconsole.data.Node;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.HostDbHostInfo;
import ru.yandex.wmconsole.data.info.TreeInfo;
import ru.yandex.wmconsole.data.info.TreeNodeInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmconsole.service.dao.TblRobotdbInfoDao;
import ru.yandex.wmconsole.service.dao.TblUrlTreesDao;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.service.IndexInfoService;
import ru.yandex.wmtools.common.util.SqlUtil;

/**
 * Created by IntelliJ IDEA.
 * User: senin
 * Date: 20.03.2007
 * Time: 10:09:24
 */
public class UrlTreeService extends TreeService {
    public static final String FIELD_HOST_ID = "host_id";
    public static final String FIELD_URL_NODE_ID = "url_node_id";
    public static final String FIELD_PARENT_URL_NODE_ID = "parent_url_node_id";
    public static final String FIELD_URL_NODE_NAME = "url_node_name";
    public static final String FIELD_URLS = "urls";
    public static final String FIELD_INDEX_COUNT = "index_count";
    public static final String FIELD_CHILDREN_COUNT = "children_count";
    public static final String FIELD_HAS_INT_LINKS = "has_int_links";
    public static final String FIELD_HAS_EXT_LINKS = "has_ext_links";
    public static final String FIELD_URLS_TREND = "urls_trend";
    public static final String FIELD_INDEX_COUNT_TREND = "index_count_trend";
    public static final String FIELD_BANNED_COUNT = "banned_count";

//    private static final String SELECT_FIND_ROOT_NODE_QUERY =
//            "SELECT " +
//                    "    id AS " + FIELD_URL_NODE_ID + " " +
//                    "FROM " +
//                    "    tbl_url_trees " +
//                    "WHERE " +
//                    "    host_id = ? " +
//                    "AND " +
//                    "    parent_id IS NULL ";

    private IndexInfoService indexInfoService;
    private IndexInfoService uaIndexInfoService;
    private IndexInfoService comIndexInfoService;
    private IndexInfoService trIndexInfoService;

    private HostDbHostInfoService hostDbHostInfoService;
    private TblRobotdbInfoDao tblRobotdbInfoDao;
    private TblUrlTreesDao tblUrlTreesDao;

//    private final DBTreeInfoCache<Long, TreeInfo> urlTreeInfoCache = new DBTreeInfoCache<Long, TreeInfo>();

    private static final ParameterizedRowMapper<TreeNodeInfo> urlNodeInfoMapper = new ParameterizedRowMapper<TreeNodeInfo>() {
        @Override
        public TreeNodeInfo mapRow(ResultSet rs, int rowNum) throws SQLException {
            long urlNodeId = rs.getLong(FIELD_URL_NODE_ID);
            Long parentUrlNodeId = SqlUtil.getLongNullable(rs, FIELD_PARENT_URL_NODE_ID);
            String urlNodeName = rs.getString(FIELD_URL_NODE_NAME);
            long urls = rs.getLong(FIELD_URLS);
            Long indexCount = SqlUtil.getLongNullable(rs, FIELD_INDEX_COUNT);
            Long bannedCount = SqlUtil.getLongNullable(rs, FIELD_BANNED_COUNT);
            if (indexCount != null && bannedCount != null && bannedCount > 0) {
                indexCount = Math.max(0, indexCount - bannedCount);
            }
            boolean hasIntLinks = rs.getBoolean(FIELD_HAS_INT_LINKS);
            boolean hasExtLinks = rs.getBoolean(FIELD_HAS_EXT_LINKS);
            Long urlsTrend = SqlUtil.getLongNullable(rs, FIELD_URLS_TREND);
            Long indexCountTrend = SqlUtil.getLongNullable(rs, FIELD_INDEX_COUNT_TREND);
            int childrenCount = rs.getInt(FIELD_CHILDREN_COUNT);
            return new TreeNodeInfo(urlNodeId, parentUrlNodeId, urlNodeName, urls, indexCount, hasIntLinks, hasExtLinks, childrenCount, urlsTrend, indexCountTrend);
        }
    };

    protected List<TreeNodeInfo> getTreeNodesList(final HostDbHostInfo hostDbHostInfo,
            final YandexSearchShard searchShard) throws InternalException
    {
        String q =
                "SELECT " +
                "    ut.id AS " + FIELD_URL_NODE_ID + ", " +
                "    ut.parent_id AS " + FIELD_PARENT_URL_NODE_ID + ", " +
                "    ut.name AS " + FIELD_URL_NODE_NAME + ", " +
                "    ut.urls AS " + FIELD_URLS + ", " +
                "    ut.index_count AS " + FIELD_INDEX_COUNT + ", " +
                "    ut.banned_count AS " + FIELD_BANNED_COUNT + ", " +
                "    ut.has_int_links AS " + FIELD_HAS_INT_LINKS + ", " +
                "    ut.has_ext_links AS " + FIELD_HAS_EXT_LINKS + ", " +
                "    ut.urls_trend AS " + FIELD_URLS_TREND + ", " +
                "    ut.index_count_trend AS " + FIELD_INDEX_COUNT_TREND + ", " +
                "    (SELECT " +
                "         count(*) " +
                "     FROM " +
                "         tbl_url_trees ut1 " +
                "     WHERE " +
                "         ut1.parent_id = ut.id " +
                "         AND ut1.shard_id = ? " +
                "    ) AS " + FIELD_CHILDREN_COUNT + " " +
                "FROM " +
                "    tbl_url_trees ut " +
                "WHERE " +
                "    ut.host_id = ? " +
                "    AND ut.shard_id = ?";

        return getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).query(
                q, urlNodeInfoMapper, searchShard.value(), hostDbHostInfo.getHostDbHostId(), searchShard.value());
    }

    public Long getHostUrlCount(final BriefHostInfo hostInfo) throws InternalException {
        HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(hostInfo.getName());
        Long result = tblRobotdbInfoDao.getUrlsByHostId(hostDbHostInfo);
        if (result == null) {
            return 0L;
        }
        return result;
    }

    /**
     * Обновляет число страниц в дереве url
     * @param treeInfo          дерево
     * @param selectedNodeId    новый выбранный узел
     */
    protected boolean updateIndexCountInTree(HostDbHostInfo hostDbHostInfo, TreeInfo treeInfo, Long selectedNodeId,
            YandexSearchShard searchShard) throws UserException, InternalException
    {
        Node root = treeInfo.getRootNode();
        treeInfo.setSelectedNodeId(selectedNodeId);
        List<Node> current = Collections.singletonList(root);
        List<Node> next = Collections.emptyList();
        boolean selectedNodeFound = false;
        boolean lastStep = false;
        boolean changed = false;
        do {
            next = new LinkedList<Node>();
            lastStep = selectedNodeFound;
            if (lastStep) {
                current = treeInfo.getSelectedNode().getChildren();
            }
            for (Node n : current) {
                if (n == treeInfo.getSelectedNode()) {
                      selectedNodeFound = true;
                }
                boolean changedNode = updateIndexCount(hostDbHostInfo, n, treeInfo.getId2node(), searchShard);
                changed = changed || changedNode;
                next.addAll(n.getChildren());
            }
            if (lastStep) {
                break;
            }
            current = next;
        } while (!next.isEmpty());
        return changed;
    }

    /**
     * Обновляет число страниц в индексе для узла дерева
     *
     * @param n
     * @param id2node
     * @return          изменен ли данный узел
     */
    protected boolean updateIndexCount(HostDbHostInfo hostDbHostInfo, Node n, Map<Long, Node> id2node,
            YandexSearchShard searchShard) throws UserException, InternalException
    {
        TreeNodeInfo info = n.getInfo();
        if (info.getIndexCount() == null) {
            final String path = calculatePath(n, id2node);
            IndexInfoService countryIndexInfoService = indexInfoService;
            switch(searchShard) {
                case UA: countryIndexInfoService = uaIndexInfoService; break;
                case COM: countryIndexInfoService = comIndexInfoService; break;
                case COM_TR: countryIndexInfoService = trIndexInfoService; break;

                default: countryIndexInfoService = indexInfoService;
            }
            long indexCount = countryIndexInfoService.getIndexCount(hostDbHostInfo.getName(), path);
            info.setIndexCount(indexCount);
            return true;
        }
        return false;
    }

    public TreeInfo getUrlTreeInfo(BriefHostInfo hostInfo, Long selectedNodeId) throws InternalException, UserException {
        HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(hostInfo.getName());
        YandexSearchShard searchShard = tblUrlTreesDao.getOptimumShardId(hostDbHostInfo);
        return getUrlTreeInfo(hostDbHostInfo, selectedNodeId, true, searchShard);
    }

    public TreeInfo getUrlTreeInfo(HostDbHostInfo hostDbHostInfo, Long selectedNodeId, boolean needUpdateIndexCount,
            YandexSearchShard searchShard) throws InternalException, UserException
    {
        Date updatedOn = getHostInfoService().getUpdatedOn(hostDbHostInfo);
//        TreeInfo cachedUrlTreeInfo = urlTreeInfoCache.search(hostDbHostInfo.getHostDbHostId(), updatedOn);
//        if (cachedUrlTreeInfo != null) {
//            cachedUrlTreeInfo.setSelectedNodeId(selectedNodeId);
//            if (needUpdateIndexCount) {
//                boolean changed = updateIndexCountInTree(hostDbHostInfo, cachedUrlTreeInfo, selectedNodeId);
//                if (changed) {
//                    urlTreeInfoCache.save(hostDbHostInfo.getHostDbHostId(), cachedUrlTreeInfo);
//                }
//            }
//            return cachedUrlTreeInfo;
//        }

        List<TreeNodeInfo> list = getTreeNodesList(hostDbHostInfo, searchShard);
        if (list.isEmpty()) {
            return null;
        }

        Map<Long, Node> id2node = new HashMap<Long, Node>();
        Node rootNode = convertListToTree(list, id2node);
        TreeInfo result = new TreeInfo(selectedNodeId, rootNode, id2node, updatedOn);
        if (needUpdateIndexCount) {
            updateIndexCountInTree(hostDbHostInfo, result, selectedNodeId, searchShard);
        }
//        urlTreeInfoCache.save(hostDbHostInfo.getHostDbHostId(), result);
        return result;
    }

    private String calculatePath(final Node current, final Map<Long, Node> idNode) {
        final List<String> stack = new LinkedList<String>();
        Node n = current;
        while (n != null) {
            TreeNodeInfo info = n.getInfo();
            stack.add(0, info.getName());
            Long parentId = info.getParentId();
            if (parentId != null) {
                n = idNode.get(parentId);
            } else {
                n = null;
            }
        }

        final StringBuilder builder = new StringBuilder();
        for (String name : stack) {
            builder.append(name);
        }
        return builder.toString();
    }

    @Override
    protected TreeNodeInfo createTreeNodeInfo(TreeNodeInfo info) {
        return new TreeNodeInfo(info);
    }

    public TreeInfo createFakeRootNode(String hostName, long urlCount, Long indexCount) {
        TreeNodeInfo nodeInfo = new TreeNodeInfo(-1, null, hostName, urlCount, indexCount, false, false, 0, null, null);
        Node rootNode = new Node(nodeInfo);
        return new TreeInfo(-1L, rootNode, new HashMap<Long, Node>(), new Date());
    }

    public Long getPreciseIndexCount(final BriefHostInfo hostInfo) throws InternalException {
        HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(hostInfo.getName());
        YandexSearchShard shard = tblUrlTreesDao.getOptimumShardId(hostDbHostInfo);
        return getPreciseIndexCount(hostDbHostInfo, shard);
    }

    public Long getPreciseIndexCount(final HostDbHostInfo hostDbHostInfo, final YandexSearchShard searchShard) throws InternalException {
        return tblUrlTreesDao.getIndexCount(hostDbHostInfo, searchShard);
    }

    @Required
    public void setIndexInfoService(IndexInfoService indexInfoService) {
        this.indexInfoService = indexInfoService;
    }

    @Required
    public void setUaIndexInfoService(IndexInfoService uaIndexInfoService) {
        this.uaIndexInfoService = uaIndexInfoService;
    }

    @Required
    public void setComIndexInfoService(IndexInfoService comIndexInfoService) {
        this.comIndexInfoService = comIndexInfoService;
    }

    @Required
    public void setTrIndexInfoService(IndexInfoService trIndexInfoService) {
        this.trIndexInfoService = trIndexInfoService;
    }

    @Required
    public void setHostDbHostInfoService(HostDbHostInfoService hostDbHostInfoService) {
        this.hostDbHostInfoService = hostDbHostInfoService;
    }

    @Required
    public void setTblRobotdbInfoDao(TblRobotdbInfoDao tblRobotdbInfoDao) {
        this.tblRobotdbInfoDao = tblRobotdbInfoDao;
    }

    @Required
    public void setTblUrlTreesDao(TblUrlTreesDao tblUrlTreesDao) {
        this.tblUrlTreesDao = tblUrlTreesDao;
    }
}
