package ru.yandex.wmconsole.service;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.transaction.TransactionStatus;
import ru.yandex.common.framework.user.UserInfo;
import ru.yandex.common.util.collections.Pair;
import ru.yandex.common.util.db.LongRowMapper;
import ru.yandex.common.util.db.StringRowMapper;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.webmaster.common.host.HostEventService;
import ru.yandex.webmaster.common.host.dao.TblUsersHostsCounterDao;
import ru.yandex.webmaster.common.host.dao.TblUsersHostsDao;
import ru.yandex.wmconsole.data.VerificationStateEnum;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.DelegationInfo;
import ru.yandex.wmconsole.data.info.FreeVerificationInfo;
import ru.yandex.wmconsole.data.info.OwnedHostInfo;
import ru.yandex.wmconsole.data.info.ShortHostInfo;
import ru.yandex.wmconsole.data.info.ShortUsersHostsInfo;
import ru.yandex.wmconsole.data.info.UsersHostsInfo;
import ru.yandex.wmconsole.data.info.UsersHostsVerificationInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmconsole.error.ClientException;
import ru.yandex.wmconsole.error.ClientProblem;
import ru.yandex.wmconsole.service.error.WMCUserProblem;
import ru.yandex.wmconsole.util.SimpleUsersHostsInfoHandler;
import ru.yandex.wmconsole.verification.AutoVerifier;
import ru.yandex.wmconsole.verification.PDDAutoVerifier;
import ru.yandex.wmconsole.verification.VerificationTypeEnum;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.service.UserService;
import ru.yandex.wmtools.common.util.CollectionUtil;
import ru.yandex.wmtools.common.util.ServiceTransactionCallbackWithoutResult;
import ru.yandex.wmtools.common.util.SqlUtil;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Created by IntelliJ IDEA.
 * User: senin
 * Date: 20.03.2007
 * Time: 10:09:24
 */
public class UsersHostsService extends AbstractDbService {
    private static final Logger log = LoggerFactory.getLogger(UsersHostsService.class);

    private static final String FIELD_STATE = "state";
    private static final String FIELD_VERIFICATION_UIN = "verification_uin";
    private static final String FIELD_VERIFICATION_TYPE = "verification_type";
    private static final String FIELD_VERIFICATION_DATE = "verified_on";
    private static final String FIELD_VERIFICATION_STATE = "verification_state";
    private static final String FIELD_VERIFY_FAULT_LOG = "verify_fault_log";
    private static final String FIELD_USER_ID = "user_id";
    private static final String FIELD_HOST_ID = "host_id";
    private static final String FIELD_HOST_NAME = "host_name";
    private static final String FIELD_CODE = "code";
    private static final String FIELD_MAIN_MIRROR_ID = "mirror_id";

    private DNSVerificationsService dnsVerificationsService;
    private UserService userService;
    private HostVisitorService hostVisitorService;
    private DelegationsService delegationsService;
    private AutoVerifier autoVerifier;
    private PDDAutoVerifier pddAutoVerifier;
    private HostInfoService hostInfoService;
    private HostEventService hostEventService;

    private TblUsersHostsDao tblUsersHostsDao;
    private TblUsersHostsCounterDao tblUsersHostsCounterDao;

    private static final String SELECT_ALL_VERIFIED_USERS_AND_HOSTS_QUERY =
            "SELECT " +
                    "   uh.user_id AS " + FIELD_USER_ID + ", " +
                    "   h.name AS " + FIELD_HOST_NAME + " " +
                    "FROM " +
                    "   tbl_users_hosts uh " +
                    "JOIN " +
                    "   tbl_hosts h " +
                    "USING(host_id) " +
                    "WHERE " +
                    "   uh.state IN (" + VerificationStateEnum.getCommaSeparatedListForVerifiedStates() + ") " +
                    "AND " +
                    "   NOT uh.verification_type = " + VerificationTypeEnum.CHEAT.value();

    private static final String SELECT_VERIFIED_USERS_AND_HOSTS_WITH_REMAINDER_QUERY =
            "SELECT " +
                    "   uh.user_id AS " + FIELD_USER_ID + ", " +
                    "   h.name AS " + FIELD_HOST_NAME + " " +
                    "FROM " +
                    "   tbl_users_hosts uh " +
                    "JOIN " +
                    "   tbl_hosts h " +
                    "USING(host_id) " +
                    "WHERE " +
                    "   uh.state IN (" + VerificationStateEnum.getCommaSeparatedListForVerifiedStates() + ") " +
                    "AND " +
                    "   uh.user_id MOD ? = ? " +
                    "AND " +
                    "   NOT uh.verification_type = " + VerificationTypeEnum.CHEAT.value();

    private static final String SELECT_VERIFIED_AND_CHEAT_USERS_AND_HOSTS_WITH_REMAINDER_QUERY =
            "SELECT " +
                    "   uh.user_id AS " + FIELD_USER_ID + ", " +
                    "   h.name AS " + FIELD_HOST_NAME + " " +
                    "FROM " +
                    "   tbl_users_hosts uh " +
                    "JOIN " +
                    "   tbl_hosts h " +
                    "USING(host_id) " +
                    "WHERE " +
                    "   uh.state IN (" + VerificationStateEnum.getCommaSeparatedListForVerifiedStates() + ") " +
                    "AND " +
                    "   uh.user_id MOD ? = ? ";

    private static final String SELECT_USERS_UNPROCESSED_HOSTS_LIST_QUERY =
            "SELECT " +
                    "   uh.state AS " + FIELD_STATE + ", " +
                    "   uh.verification_uin AS " + FIELD_VERIFICATION_UIN + ", " +
                    "   uh.verification_type AS " + FIELD_VERIFICATION_TYPE + ", " +
                    "   uh.verified_on AS " + FIELD_VERIFICATION_DATE + ", " +
                    "   uh.verify_fault_log AS " + FIELD_VERIFY_FAULT_LOG + ", " +
                    "   uh.user_id AS " + FIELD_USER_ID + ", " +
                    "   uh.host_id AS " + FIELD_HOST_ID + ", " +
                    "   h.name AS " + FIELD_HOST_NAME + " " +
                    "FROM " +
                    "   tbl_users_hosts uh, " +
                    "   tbl_hosts h " +
                    "WHERE " +
                    "   uh.host_id = h.host_id " +
                    " AND uh.state IN (%1$s) " +
                    " AND uh.host_id IN (%2$s) ";

    private static final String SELECT_USERS_HOSTS_INFO_QUERY =
            "SELECT " +
                    "   uh.state AS " + FIELD_STATE + ", " +
                    "   uh.verification_uin AS " + FIELD_VERIFICATION_UIN + ", " +
                    "   uh.verification_type AS " + FIELD_VERIFICATION_TYPE + ", " +
                    "   uh.verified_on AS " + FIELD_VERIFICATION_DATE + ", " +
                    "   uh.verify_fault_log AS " + FIELD_VERIFY_FAULT_LOG + ", " +
                    "   uh.user_id AS " + FIELD_USER_ID + ", " +
                    "   uh.host_id AS " + FIELD_HOST_ID + ", " +
                    "   h.name AS " + FIELD_HOST_NAME + ", " +
                    "   code.code AS " + FIELD_CODE + " " +
                    "FROM " +
                    "   tbl_users_hosts uh " +
                    "LEFT JOIN " +
                    "   tbl_verification_problem_http code " +
                    "USING " +
                    "   (host_id, user_id) " +
                    "JOIN " +
                    "   tbl_hosts h " +
                    "ON uh.host_id = h.host_id " +
                    "WHERE " +
                    "   uh.user_id = ? " +
                    "AND " +
                    "   uh.host_id = ? " +
                    "";

    private static final String SELECT_VERIFIED_USERS_HOSTS_INFO_QUERY =
            "SELECT " +
                    "   uh.state AS " + FIELD_STATE + ", " +
                    "   uh.verification_uin AS " + FIELD_VERIFICATION_UIN + ", " +
                    "   uh.verification_type AS " + FIELD_VERIFICATION_TYPE + ", " +
                    "   uh.verified_on AS " + FIELD_VERIFICATION_DATE + ", " +
                    "   uh.verify_fault_log AS " + FIELD_VERIFY_FAULT_LOG + ", " +
                    "   uh.user_id AS " + FIELD_USER_ID + ", " +
                    "   uh.host_id AS " + FIELD_HOST_ID + ", " +
                    "   h.name AS " + FIELD_HOST_NAME + " " +
                    "FROM " +
                    "   tbl_users_hosts uh, " +
                    "   tbl_hosts h " +
                    "WHERE " +
                    "   uh.host_id = ? " +
                    "AND " +
                    "    state IN (" + VerificationStateEnum.getCommaSeparatedListForVerifiedStates() + ") " +
                    "AND " +
                    "   uh.host_id = h.host_id ";

    private static final String SELECT_USERS_FOR_HOST_INFO_QUERY =
            "SELECT " +
                    "   uh.state AS " + FIELD_STATE + ", " +
                    "   uh.verification_uin AS " + FIELD_VERIFICATION_UIN + ", " +
                    "   uh.verification_type AS " + FIELD_VERIFICATION_TYPE + ", " +
                    "   uh.verified_on AS " + FIELD_VERIFICATION_DATE + ", " +
                    "   uh.verify_fault_log AS " + FIELD_VERIFY_FAULT_LOG + ", " +
                    "   uh.user_id AS " + FIELD_USER_ID + ", " +
                    "   uh.host_id AS " + FIELD_HOST_ID + ", " +
                    "   h.name AS " + FIELD_HOST_NAME + " " +
                    "FROM " +
                    "   tbl_users_hosts uh, " +
                    "   tbl_hosts h " +
                    "WHERE " +
                    "   uh.host_id = ? " +
                    "AND " +
                    "   uh.host_id = h.host_id ";

    private static final String SELECT_HOSTS_VERIFIED_BY_USER_NOT_CHEAT_QUERY =
            "SELECT " +
                    "    uh.host_id AS " + FIELD_HOST_ID + ", " +
                    "    h.name AS " + FIELD_HOST_NAME + ", " +
                    "    uh.state AS " + FIELD_VERIFICATION_STATE + ", " +
                    "    uh.verification_type AS " + FIELD_VERIFICATION_TYPE + " " +
                    "FROM (" +
                    "    tbl_users_hosts uh " +
                    "LEFT JOIN " +
                    "    tbl_hosts h ON (uh.host_id = h.host_id)) " +
                    "WHERE " +
                    "    uh.user_id = ?" +
                    " AND" +
                    "   uh.state IN (" + VerificationStateEnum.getCommaSeparatedListForVerifiedStates() + ") " +
                    " AND " +
                    "   NOT uh.verification_type = " + VerificationTypeEnum.CHEAT.value();

    private static final String SELECT_HOSTS_VERIFIED_BY_USER_QUERY =
            "SELECT " +
                    "    uh.host_id AS " + FIELD_HOST_ID + ", " +
                    "    h.name AS " + FIELD_HOST_NAME + ", " +
                    "    uh.state AS " + FIELD_VERIFICATION_STATE + ", " +
                    "    uh.verification_type AS " + FIELD_VERIFICATION_TYPE + " " +
                    "FROM (" +
                    "    tbl_users_hosts uh " +
                    "LEFT JOIN " +
                    "    tbl_hosts h ON (uh.host_id = h.host_id)) " +
                    "WHERE " +
                    "    uh.user_id = ?" +
                    " AND" +
                    "   uh.state IN (" + VerificationStateEnum.getCommaSeparatedListForVerifiedStates() + ") ";


    private static final String SELECT_CHECK_HOSTS_VERIFIED_BY_USER_QUERY =
            "SELECT " +
                    "    uh.host_id AS " + FIELD_HOST_ID + ", " +
                    "    h.name AS " + FIELD_HOST_NAME + ", " +
                    "    uh.state AS " + FIELD_VERIFICATION_STATE + ", " +
                    "    uh.verification_type AS " + FIELD_VERIFICATION_TYPE + " " +
                    "FROM (" +
                    "    tbl_users_hosts uh " +
                    "LEFT JOIN " +
                    "    tbl_hosts h ON (uh.host_id = h.host_id)) " +
                    "WHERE " +
                    "    uh.user_id = ?" +
                    " AND" +
                    "   h.name IN (%1$s) " +
                    " AND " +
                    "   NOT uh.verification_type = " + VerificationTypeEnum.CHEAT.value();

    private static final String SELECT_GET_HOSTS_VERIFIED_BY_USER_QUERY =
            "SELECT " +
                    "    uh.host_id AS " + FIELD_HOST_ID + ", " +
                    "    h.name AS " + FIELD_HOST_NAME + ", " +
                    "    uh.state AS " + FIELD_VERIFICATION_STATE + ", " +
                    "    uh.verification_type AS " + FIELD_VERIFICATION_TYPE + ", " +
                    "    h.mirror_id AS " + FIELD_MAIN_MIRROR_ID + " " +
                    "FROM (" +
                    "    tbl_users_hosts uh " +
                    "LEFT JOIN " +
                    "    tbl_hosts h ON (uh.host_id = h.host_id)) " +
                    "WHERE " +
                    "    uh.user_id = ?";

    private static final String SELECT_VERIFICATIONS_PAGE =
            "SELECT id, " +
                    "   uh.state AS " + FIELD_STATE + ", " +
                    "   uh.verification_uin AS " + FIELD_VERIFICATION_UIN + ", " +
                    "   uh.verification_type AS " + FIELD_VERIFICATION_TYPE + ", " +
                    "   uh.verified_on AS " + FIELD_VERIFICATION_DATE + ", " +
                    "   uh.verify_fault_log AS " + FIELD_VERIFY_FAULT_LOG + ", " +
                    "   uh.user_id AS " + FIELD_USER_ID + ", " +
                    "   0 AS " + FIELD_HOST_ID + ", " +
                    "   uh.host_name AS " + FIELD_HOST_NAME + " " +
                    "FROM " +
                    "   tmp_users_hosts uh " +
                    "WHERE id > ? " +
                    "ORDER BY id asc " +
                    "LIMIT ?";

    private static final String DELETE_VERIFICATION_PROBLEM_HTTP_QUERY =
            "DELETE FROM tbl_verification_problem_http WHERE user_id = ? AND host_id = ?";

    private static final String REPLACE_VERIFICATION_PROBLEM_HTTP_QUERY =
            "REPLACE INTO tbl_verification_problem_http (user_id, host_id, code) VALUES (?, ?, ?)";

    private static final String SELECT_MIRRORS_COUNT_QUERY =
            "SELECT COUNT(uh.host_id) FROM tbl_users_hosts uh JOIN tbl_hosts h ON (uh.host_id = h.host_id) WHERE uh.user_id = ? AND h.mirror_id = ?";

    public List<UsersHostsInfo> getUsersHostsInfoList(List<Long> hostIds) throws InternalException {
        if (hostIds == null || hostIds.size() == 0) {
            return Collections.emptyList();
        }

        String queryString = String.format(SELECT_USERS_UNPROCESSED_HOSTS_LIST_QUERY, VerificationStateEnum.getCommaSeparatedListForVerifiedStates(), SqlUtil.getCommaSeparatedList(hostIds));

        return getJdbcTemplate(WMCPartition.nullPartition()).query(queryString, fastUsersHostsInfoRowMapper);
    }

    public UsersHostsInfo getUsersHostsInfo(long userId, long hostId) throws InternalException {
        List<UsersHostsInfo> verInfoList = getJdbcTemplate(new WMCPartition(null, userId)).query(
                SELECT_USERS_HOSTS_INFO_QUERY, usersHostsInfoRowMapperWithHttpCode, userId, hostId);
        if ((verInfoList == null) || verInfoList.isEmpty()) {
            return null;
        }
        // NOTE: size of verInfoList cannot be > 1 because hostId and userId is private key
        return verInfoList.get(0);
    }

    public void putHostInVerificationQueue(long userId, long hostId, VerificationTypeEnum verType)
            throws ClientException, InternalException {
        log.debug("Update verification state: userId = " + userId + ", hostId = " + hostId);
        UsersHostsInfo usersHostsInfo = getUsersHostsInfo(userId, hostId);
        if (usersHostsInfo == null) {
            throw new ClientException(ClientProblem.HOST_NOT_OWNED_BY_USER, "Host is not added to the users host list");
        }

        tblUsersHostsDao.updateVerificationInfo(userId, hostId, verType);

        hostVisitorService.fireCheck();
    }

    public void updateVerificationInfoInTransaction(final UsersHostsInfo info) throws InternalException {
        try {
            getServiceTransactionTemplate(WMCPartition.nullPartition()).executeInService(
                    new ServiceTransactionCallbackWithoutResult() {
                        @Override
                        protected void doInTransactionWithoutResult(TransactionStatus transactionStatus)
                                throws InternalException {
                            // Сохраняем информацию о подтверждении сайта
                            updateHostVerificationInfo(info);
                            // Сохраняем код ошибки
                            updateVerificationProblemHttpCode(info.getUserId(), info.getHostId(), info.getHttpCode());
                            if (VerificationStateEnum.VERIFIED == info.getVerificationState()) {
                                descreaseHostsCounter(info.getUserId());
                            }
                        }
                    });
        } catch (UserException e) {
            throw new AssertionError("unreachable");
        }
    }

    public List<UsersHostsInfo> listVerifiedUsersForHost(long hostId) throws InternalException {
        return CollectionUtil.removeNulls(getJdbcTemplate(WMCPartition.nullPartition()).query(
                SELECT_VERIFIED_USERS_HOSTS_INFO_QUERY, usersHostsInfoRowMapper, hostId));
    }

    public UsersHostsVerificationInfo getUsersHostsVerificationInfo(long hostId, long userId) throws InternalException {
        UsersHostsInfo usersHostsInfo = getUsersHostsInfo(userId, hostId);

        if (usersHostsInfo == null) {
            DelegationInfo delegationInfo = delegationsService.getDelegationInfoForHostAndUser(hostId, userId);
            if (delegationInfo == null) {
                return null;
            }
            return new UsersHostsVerificationInfo(delegationInfo, userService.getUserInfo(delegationInfo.getUserIdGave()));
        }
        if (VerificationTypeEnum.DELEGATION.equals(usersHostsInfo.getVerificationType())) {
            DelegationInfo delegationInfo = delegationsService.getDelegationInfoForHostAndUser(hostId, userId);
            if (delegationInfo != null) {
                return new UsersHostsVerificationInfo(usersHostsInfo,
                        delegationInfo,
                        userService.getUserInfo(delegationInfo.getUserIdGave()));
            }
        }
        return new UsersHostsVerificationInfo(usersHostsInfo);
    }

    public List<UsersHostsVerificationInfo> listVerificationsAndDelegationsForHost(long hostId) throws InternalException, UserException {
        List<DelegationInfo> delegations = delegationsService.listDelegationsForHost(hostId);
        Map<Pair<Long, Long>, DelegationInfo> delegationsByUIDAndHID = new HashMap<Pair<Long, Long>, DelegationInfo>();
        for (DelegationInfo delegationInfo : delegations) {
            delegationsByUIDAndHID.put(
                    new Pair<Long, Long>(delegationInfo.getUserGotInfo().getUserId(), delegationInfo.getHostId()), delegationInfo);
        }

        List<UsersHostsVerificationInfo> result = new ArrayList<UsersHostsVerificationInfo>();

        List<UsersHostsInfo> usersHosts = listVerifiedUsersForHost(hostId);
        for (UsersHostsInfo usersHostsInfo : usersHosts) {
            if (usersHostsInfo.getVerificationState().isVerified()) {
                Pair<Long, Long> userAndHost = new Pair<Long, Long>(usersHostsInfo.getUserId(), usersHostsInfo.getHostId());
                DelegationInfo delegationInfo = delegationsByUIDAndHID.get(userAndHost);
                if (delegationInfo == null) {
                    //Host verified, not with delegation
                    result.add(new UsersHostsVerificationInfo(usersHostsInfo));
                } else {
                    //Host delegated and added - removing this user-host pair
                    delegationsByUIDAndHID.remove(userAndHost);
                    result.add(new UsersHostsVerificationInfo(usersHostsInfo, delegationInfo, userService.getUserInfo(delegationInfo.getUserIdGave())));
                }
            }
        }

        //Host was delegated. Not removed pairs
        for (DelegationInfo delegationInfo : delegationsByUIDAndHID.values()) {
            result.add(new UsersHostsVerificationInfo(delegationInfo, userService.getUserInfo(delegationInfo.getUserIdGave())));
        }

        for (Iterator<UsersHostsVerificationInfo> it = result.iterator(); it.hasNext(); ) {
            if (it.next().getUserGotInfo() == null) {
                it.remove();
            }
        }
        return result;
    }

    public List<UsersHostsInfo> listUsersForHost(long hostId) throws InternalException {
        return CollectionUtil.removeNulls(getJdbcTemplate(WMCPartition.nullPartition()).query(
                SELECT_USERS_FOR_HOST_INFO_QUERY, usersHostsInfoRowMapper, hostId));
    }

    public List<ShortUsersHostsInfo> listUsersForHostShortInfo(long hostId) throws InternalException {
        return CollectionUtil.removeNulls(tblUsersHostsDao.listUsersForHostShortInfo(hostId));
    }

    @Deprecated
    public void reverifyAll(long hostId) throws InternalException {
        int rowNum = tblUsersHostsDao.reverifyAll(hostId);
        log.debug("Reverify all: " + rowNum + " rows updated.");
        hostVisitorService.fireCheck();
    }

    public void cancelAnyVerification(long hostId, long userId) throws InternalException, UserException {
        delegationsService.deleteAllDelegationsGivenToUserForHost(userId, hostId, userId);
        dnsVerificationsService.removeFromDNSVerificationsTable(hostId, userId);
        changeVerificationState(hostId, userId, VerificationStateEnum.CANCELLED);
    }

    public List<Long> getUserMainMirrorHostIds(long userId) throws InternalException {
        String q = "SELECT uh.host_id FROM tbl_users_hosts uh JOIN tbl_hosts h ON uh.host_id = h.host_id " +
                "WHERE uh.user_id = ? AND h.mirror_id IS NULL";
        return getJdbcTemplate(WMCPartition.nullPartition()).query(q, new LongRowMapper(), userId);
    }

    public List<Long> getHostVerifiedUserIds(Long hostId) throws InternalException {
        return tblUsersHostsDao.getVerifiedUserByHosts(hostId);
    }

    public List<BriefHostInfo> getVerifiedMirrorsForHostByUser(final BriefHostInfo mainMirror, final long userId) throws InternalException {
        String q = "SELECT h.host_id, h.name, h.mirror_id FROM tbl_users_hosts uh JOIN tbl_hosts h ON uh.host_id = h.host_id " +
                "WHERE uh.user_id = ? AND h.mirror_id = ? AND state IN (" + VerificationStateEnum.getCommaSeparatedListForVerifiedStates() + ") ";
        final ParameterizedRowMapper<BriefHostInfo> m = new ParameterizedRowMapper<BriefHostInfo>() {
            @Override
            public BriefHostInfo mapRow(ResultSet rs, int rowNum) throws SQLException {
                return new BriefHostInfo(rs.getLong("host_id"), rs.getString("name"), SqlUtil.getLongNullable(rs, "mirror_id"));
            }
        };
        return getJdbcTemplate(WMCPartition.nullPartition()).query(q, m, userId, mainMirror.getId());
    }

    public void assertUserMayRedelegateHost(long userId, long hostId) throws UserException, InternalException {
        UsersHostsInfo usersHostsInfo = getUsersHostsInfo(userId, hostId);
        if (usersHostsInfo == null) {
            throw new UserException(WMCUserProblem.HOST_NOT_OWNED_BY_USER, "This host is NOT added to the list.");
        }

        if (!usersHostsInfo.getVerificationState().isVerified()) {
            throw new UserException(WMCUserProblem.HOST_NOT_VERIFIED_BY_USER, "Host " + hostId + " is not verified by user " + userId);
        }

        if (VerificationTypeEnum.CHEAT.equals(usersHostsInfo.getVerificationType())) {
            throw new UserException(WMCUserProblem.DELEGATION_FORBIDDEN,
                    "Delegation of host " + hostId + " may not by given by user " + userId + " (he is cheater!)");
        }

        // check, if user has permissions to delegate.
        // The only case, when he doesn't - permissions were delegated to him with no option "may redelegate"
        if (VerificationTypeEnum.DELEGATION.equals(usersHostsInfo.getVerificationType())) {
            DelegationInfo delegInfo = delegationsService.getDelegationInfoForHostAndUser(usersHostsInfo.getHostId(), usersHostsInfo.getUserId());
            if (!delegInfo.isMayRedelegate()) {
                throw new UserException(WMCUserProblem.DELEGATION_FORBIDDEN, "Delegation of host " + hostId + " may not by given by user " + userId);
            }
        }
    }

    public Collection<FreeVerificationInfo> getFreeAlreadyVerifiedHosts(long userId) throws InternalException {
        UserInfo userInfo = userService.getUserInfo(userId);
        if (userInfo == null) {
            return null;
        }
        // 1. hosts, eligible for auto-verification
        Collection<FreeVerificationInfo> res = autoVerifier.getHostsApplicableForAutoVerify(userInfo);

        // 2. delegated hosts
        Collection<FreeVerificationInfo> addOn = delegationsService.getFreeDelegationsForUser(userId);
        mergeAddOnToRes(res, addOn);

        // 3. hosts, that are already verified in other services
        addOn = pddAutoVerifier.getHostsApplicableForAutoVerify(userService.getUserInfo(userId));
        mergeAddOnToRes(res, addOn);

        // exclude hosts, that are already added by user
        List<ShortHostInfo> exclude = hostInfoService.getShortHostList(userId);
        for (ShortHostInfo info : exclude) {
            res.remove(new FreeVerificationInfo(info.getName()));
        }

        return res;
    }

    private void mergeAddOnToRes(Collection<FreeVerificationInfo> res, Collection<FreeVerificationInfo> addOn) {
        for (FreeVerificationInfo info : addOn) {
            boolean found = false;
            for (FreeVerificationInfo elem : res) {
                if (info.equals(elem)) {
                    elem.addVerificationType(VerificationTypeEnum.DELEGATION);
                    found = true;
                    break;
                }
            }
            if (!found) {
                res.add(info);
            }
        }
    }

    public void getFullUsersHostsInfo(final SimpleUsersHostsInfoHandler handler) throws InternalException {
        getJdbcTemplate(WMCPartition.nullPartition())
                .query(SELECT_ALL_VERIFIED_USERS_AND_HOSTS_QUERY,
                        new RowCallbackHandler() {
                            @Override
                            public void processRow(ResultSet resultSet) throws SQLException {
                                handler.handleInfo(resultSet.getString(FIELD_HOST_NAME),
                                        resultSet.getLong(FIELD_USER_ID));
                            }
                        });
    }

    /**
     * Получить подтвержденные хосты для пользователей c <code></>user_id % module == remainder</code>
     *
     * @param handler
     * @param module
     * @param remainder
     * @throws InternalException
     */
    public void getFullUsersHostsInfoIncludingCheat(final SimpleUsersHostsInfoHandler handler, long module, long remainder) throws InternalException {
        getJdbcTemplate(WMCPartition.nullPartition())
                .query(SELECT_VERIFIED_AND_CHEAT_USERS_AND_HOSTS_WITH_REMAINDER_QUERY,
                        new RowCallbackHandler() {
                            @Override
                            public void processRow(ResultSet resultSet) throws SQLException {
                                handler.handleInfo(resultSet.getString(FIELD_HOST_NAME),
                                        resultSet.getLong(FIELD_USER_ID));
                            }
                        },
                        module,
                        remainder);
    }

    /**
     * Получить подтвержденные хосты для пользователей c <code></>user_id % module == remainder</code>
     *
     * @param handler
     * @param module
     * @param remainder
     * @throws InternalException
     */
    public void getFullUsersHostsInfo(final SimpleUsersHostsInfoHandler handler, long module, long remainder) throws InternalException {
        getJdbcTemplate(WMCPartition.nullPartition())
                .query(SELECT_VERIFIED_USERS_AND_HOSTS_WITH_REMAINDER_QUERY,
                        new RowCallbackHandler() {
                            @Override
                            public void processRow(ResultSet resultSet) throws SQLException {
                                handler.handleInfo(resultSet.getString(FIELD_HOST_NAME),
                                        resultSet.getLong(FIELD_USER_ID));
                            }
                        },
                        module,
                        remainder);
    }

    /**
     * Returns brief information about each host verified by given user. CHEAT verifications are excluded.
     *
     * @param userId User, for which to get host information.
     * @return Returns brief information about each host of given user.
     * @throws InternalException if db error occurs
     */
    public List<ShortHostInfo> getHostsVerifiedByUser(Long userId) throws InternalException {
        List<ShortHostInfo> result = getJdbcTemplate(WMCPartition.nullPartition()).query(
                SELECT_HOSTS_VERIFIED_BY_USER_NOT_CHEAT_QUERY, shortHostInfoMapper, userId);
        return result;
    }

    /**
     * Returns brief information only for hosts verified by given user. CHEAT verifications are excluded.
     *
     * @param userId User, for which to get host information.
     * @return Returns brief information about each host of given user.
     * @throws InternalException if db error occurs
     */
    public List<ShortHostInfo> checkHostsVerifiedByUser(Long userId, String[] hosts) throws InternalException {
        String query = String.format(SELECT_CHECK_HOSTS_VERIFIED_BY_USER_QUERY, SqlUtil.createQuestionMarks(hosts.length));
        final Object[] params = new Object[1 + hosts.length];
        params[0] = userId;
        System.arraycopy(hosts, 0, params, 1, hosts.length);
        List<ShortHostInfo> result = getJdbcTemplate(WMCPartition.nullPartition()).query(query, shortHostInfoMapper, params);
        return result;
    }

    /**
     * Returns brief information for all hosts of given user.
     *
     * @param userId User, for which to get host information.
     * @param showCheat list cheat verifications
     * @return Returns brief information about each host of given user.
     * @throws InternalException if db error occurs
     */
    public List<OwnedHostInfo> getUserHosts(Long userId, boolean showCheat) throws InternalException {
        String q = SELECT_GET_HOSTS_VERIFIED_BY_USER_QUERY;
        if (!showCheat) {
            q += " AND NOT uh.verification_type = " + VerificationTypeEnum.CHEAT.value();
        }
        List<OwnedHostInfo> result = getJdbcTemplate(WMCPartition.nullPartition())
                .query(q, ownedHostInfoMapper, userId);
        return result;
    }

    /**
     * Returns brief information about each host verified by given user. CHEAT verifications are included.
     *
     * @param userId User, for which to get host information.
     * @return Returns brief information about each host of given user.
     * @throws InternalException if db error occurs
     */
    public List<ShortHostInfo> getAllHostsVerifiedByUser(Long userId) throws InternalException {
        List<ShortHostInfo> result = getJdbcTemplate(WMCPartition.nullPartition()).query(
                SELECT_HOSTS_VERIFIED_BY_USER_QUERY, shortHostInfoMapper, userId);
        return result;
    }

    /**
     * Inserts or deletes http code explaining verification problem
     *
     * @param userId   User, which tries to verify host
     * @param hostId   verifying host
     * @param httpCode http response status code, null to reset http code
     */
    public void updateVerificationProblemHttpCode(final long userId, final long hostId, final Integer httpCode) throws InternalException {
        if (httpCode == null) {
            getJdbcTemplate(WMCPartition.nullPartition()).update(
                    DELETE_VERIFICATION_PROBLEM_HTTP_QUERY,
                    userId, hostId);
        } else {
            getJdbcTemplate(WMCPartition.nullPartition()).update(
                    REPLACE_VERIFICATION_PROBLEM_HTTP_QUERY,
                    userId,
                    hostId,
                    httpCode);
        }
    }

    public long getHostsCounter(long userId) throws InternalException {
        List<Long> counter = tblUsersHostsCounterDao.getCount(userId);
        return counter.size() > 0 ? counter.iterator().next() : 0L;
    }

    public void increaseHostsCounter(long userId, long hostsCounter, int delta) throws InternalException {
        if (hostsCounter > 0) {
            tblUsersHostsCounterDao.increaseCounter(userId, delta);
        } else {
            tblUsersHostsCounterDao.insertOrIncreaseCounter(userId, delta);
        }
    }

    public void descreaseHostsCounter(long userId) throws InternalException {
        final long hostsCount = getHostsCounter(userId);
        if (hostsCount > 0) {
            tblUsersHostsCounterDao.decreaseCounter(userId);
        } else {
            tblUsersHostsCounterDao.insertOrDecreaseCounter(userId);
        }
    }

    public int getMirrorsCount(long userId, long mainMirrorId) throws InternalException {
        return getJdbcTemplate(WMCPartition.nullPartition()).queryForInt(SELECT_MIRRORS_COUNT_QUERY, userId, mainMirrorId);
    }

    public void resetVerificationsToInProgressForHost(UsersHostsInfo oldHostsInfo) throws InternalException {
        final VerificationStateEnum newState = VerificationStateEnum.RECHECK_WAITING == oldHostsInfo.getVerificationState()
                ? VerificationStateEnum.RECHECK_IN_PROGRESS
                : VerificationStateEnum.IN_PROGRESS;
        final UsersHostsInfo newInfo = new UsersHostsInfo(
                newState,
                oldHostsInfo.getVerificationUin(),
                oldHostsInfo.getVerificationType(),
                new Date(),
                oldHostsInfo.getVerifyFaultLog(),
                oldHostsInfo.getUserId(),
                oldHostsInfo.getHostId(),
                oldHostsInfo.getHostName(),
                oldHostsInfo.getUserInfo(),
                oldHostsInfo.getHttpCode());

        int ret = tblUsersHostsDao.updateVerificationStateAndTypeByState(newInfo, oldHostsInfo.getVerificationState());
        if (ret == 0) {
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "Wrong previous verification state");
        }
        addHostVerifiedEvent(newInfo.getHostId(), newInfo.getUserId(), newInfo.getUserId(), newState);
    }

    protected void updateHostVerificationInfo(final UsersHostsInfo info) throws InternalException {
        tblUsersHostsDao.updateVerificationStatAndType(info);
        addHostVerifiedEvent(info.getHostId(), info.getUserId(), info.getUserId(), info.getVerificationState());
    }

    public void changeVerificationState(long hostId, long userId, VerificationStateEnum newState) throws InternalException {
        tblUsersHostsDao.updateVerificationState(userId, hostId, newState);
        log.debug("Host " + hostId + " is put into state " + newState + " for user " + userId + ".");
        addHostVerifiedEvent(hostId, userId, userId, newState);
    }

    protected void addHostVerifiedEvent(long hostId, long forUserId, long byUserId, VerificationStateEnum state) throws InternalException {
        if (state != VerificationStateEnum.VERIFIED) {
            return;
        }
        try {
            hostEventService.hostIsVerified(hostId, forUserId, byUserId);
        } catch (Exception e) {
            log.error(
                    "Unable to save host verified event: hostId=" + hostId + " forUserId=" + forUserId + " byUserId=" + byUserId + " state=" + state);
        }
    }

    public List<UserInfo> getVerifiedUserForHost(BriefHostInfo briefHostInfo, boolean includeCheat) throws InternalException {
        return getVerifiedUserForHost(briefHostInfo.getId(), includeCheat);
    }

    public List<UserInfo> getVerifiedUserForHost(long hostId, boolean includeCheat) throws InternalException {
        SqlCondition stateC = TblUsersHostsDao.verificationStateCondition(VerificationStateEnum.getVerifiedStates());
        SqlCondition typeC;
        if (includeCheat) {
            typeC = SqlCondition.trueCondition();
        } else {
            typeC = TblUsersHostsDao.verificationTypeCondition(EnumSet.of(VerificationTypeEnum.CHEAT)).not();
        }

        List<Long> userIds = tblUsersHostsDao.getUsersForHost(hostId, stateC.and(typeC));
        List<UserInfo> users = new ArrayList<>(userIds.size());
        for (Long userId : userIds) {
            UserInfo userInfo = userService.getUserInfo(userId);
            if (userInfo != null) {
                users.add(userInfo);
            }
        }
        return users;
    }

    public List<UsersHostsInfo> findUserHostInfo(BriefHostInfo briefHostInfo, Set<Long> userId) throws InternalException {
        SqlCondition c = SqlCondition.all(
                TblUsersHostsDao.hostId(briefHostInfo),
                TblUsersHostsDao.userId(userId)
        );

        return tblUsersHostsDao.findUserHostInfo(c, SqlLimits.first(1));
    }

    public List<String> getHostsAddedByUsers(int hostIdDivisor, int hostIdModule) throws InternalException {
        if (hostIdModule >= hostIdDivisor || hostIdModule < 0) {
            throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "host module out of rage");
        }

        return getJdbcTemplate(WMCPartition.nullPartition()).
                query("SELECT h.name FROM tbl_hosts h JOIN tbl_users_hosts uh ON (h.host_id = uh.host_id) WHERE uh.host_id MOD ? = ? GROUP BY uh.host_id HAVING COUNT(user_id) > 0", new StringRowMapper(), hostIdDivisor, hostIdModule);
    }

    public List<Pair<UsersHostsInfo, Long>> listAllVerifications(int page, int pageSize) throws InternalException {
        return getJdbcTemplate(WMCPartition.nullPartition())
                .query(SELECT_VERIFICATIONS_PAGE,
                        new ParameterizedRowMapper<Pair<UsersHostsInfo, Long>>() {
                            @Override
                            public Pair<UsersHostsInfo, Long> mapRow(ResultSet resultSet, int i) throws SQLException {
                                return Pair.of(fastUsersHostsInfoRowMapper.mapRow(resultSet, i), resultSet.getLong("id"));
                            }
                        }
                        , page, pageSize);
    }

    private static final ParameterizedRowMapper<ShortHostInfo> shortHostInfoMapper = new ParameterizedRowMapper<ShortHostInfo>() {
        @Override
        public ShortHostInfo mapRow(ResultSet resultSet, int rowNumber) throws SQLException {
            return new ShortHostInfo(
                    resultSet.getLong(FIELD_HOST_ID),
                    resultSet.getString(FIELD_HOST_NAME),
                    VerificationStateEnum.R.fromValueOrNull(resultSet.getInt(FIELD_VERIFICATION_STATE)),
                    VerificationTypeEnum.R.fromValueOrNull(resultSet.getInt(FIELD_VERIFICATION_TYPE))
            );
        }
    };

    private static final ParameterizedRowMapper<OwnedHostInfo> ownedHostInfoMapper = new ParameterizedRowMapper<OwnedHostInfo>() {
        @Override
        public OwnedHostInfo mapRow(ResultSet resultSet, int rowNumber) throws SQLException {
            return new OwnedHostInfo(
                    resultSet.getLong(FIELD_HOST_ID),
                    resultSet.getString(FIELD_HOST_NAME),
                    VerificationStateEnum.R.fromValueOrNull(resultSet.getInt(FIELD_VERIFICATION_STATE)),
                    VerificationTypeEnum.R.fromValueOrNull(resultSet.getInt(FIELD_VERIFICATION_TYPE)),
                    SqlUtil.getLongNullable(resultSet, FIELD_MAIN_MIRROR_ID)
            );
        }
    };

    private final ParameterizedRowMapper<UsersHostsInfo> usersHostsInfoRowMapper = new ParameterizedRowMapper<UsersHostsInfo>() {
        @Override
        public UsersHostsInfo mapRow(ResultSet rs, int i) throws SQLException {
            long userId = rs.getLong(FIELD_USER_ID);
            UserInfo userInfo = userService.getUserInfo(userId);
            if (userInfo == null) {
                return null;
            }
            return new UsersHostsInfo(
                    VerificationStateEnum.R.fromValueOrNull(rs.getInt(FIELD_STATE)),
                    rs.getBigDecimal(FIELD_VERIFICATION_UIN).longValue(),
                    VerificationTypeEnum.R.fromValueOrNull(rs.getInt(FIELD_VERIFICATION_TYPE)),
                    SqlUtil.safeGetTimestamp(rs, FIELD_VERIFICATION_DATE),
                    rs.getString(FIELD_VERIFY_FAULT_LOG),
                    userId,
                    rs.getLong(FIELD_HOST_ID),
                    rs.getString(FIELD_HOST_NAME),
                    userInfo,
                    null
            );
        }
    };

    private final ParameterizedRowMapper<UsersHostsInfo> usersHostsInfoRowMapperWithHttpCode = new ParameterizedRowMapper<UsersHostsInfo>() {
        @Override
        public UsersHostsInfo mapRow(ResultSet rs, int i) throws SQLException {
            long userId = rs.getLong(FIELD_USER_ID);
            UserInfo userInfo = userService.getUserInfo(userId);
            if (userInfo == null) {
                return null;
            }
            return new UsersHostsInfo(
                    VerificationStateEnum.R.fromValueOrNull(rs.getInt(FIELD_STATE)),
                    rs.getBigDecimal(FIELD_VERIFICATION_UIN).longValue(),
                    VerificationTypeEnum.R.fromValueOrNull(rs.getInt(FIELD_VERIFICATION_TYPE)),
                    SqlUtil.safeGetTimestamp(rs, FIELD_VERIFICATION_DATE),
                    rs.getString(FIELD_VERIFY_FAULT_LOG),
                    userId,
                    rs.getLong(FIELD_HOST_ID),
                    rs.getString(FIELD_HOST_NAME),
                    userInfo,
                    SqlUtil.getIntNullable(rs, FIELD_CODE)
            );
        }
    };

    private static final ParameterizedRowMapper<UsersHostsInfo> fastUsersHostsInfoRowMapper = new ParameterizedRowMapper<UsersHostsInfo>() {
        @Override
        public UsersHostsInfo mapRow(ResultSet rs, int i) throws SQLException {
            long userId = rs.getLong(FIELD_USER_ID);
            return new UsersHostsInfo(
                    VerificationStateEnum.R.fromValueOrNull(rs.getInt(FIELD_STATE)),
                    rs.getBigDecimal(FIELD_VERIFICATION_UIN).longValue(),
                    VerificationTypeEnum.R.fromValueOrNull(rs.getInt(FIELD_VERIFICATION_TYPE)),
                    SqlUtil.safeGetTimestamp(rs, FIELD_VERIFICATION_DATE),
                    rs.getString(FIELD_VERIFY_FAULT_LOG),
                    userId,
                    rs.getLong(FIELD_HOST_ID),
                    rs.getString(FIELD_HOST_NAME),
                    null,
                    null
            );
        }
    };

    @Required
    public void setDnsVerificationsService(DNSVerificationsService dnsVerificationsService) {
        this.dnsVerificationsService = dnsVerificationsService;
    }

    @Required
    public void setUserService(UserService userService) {
        this.userService = userService;
    }

    @Required
    public void setHostVisitorService(HostVisitorService hostVisitorService) {
        this.hostVisitorService = hostVisitorService;
    }

    @Required
    public void setDelegationsService(DelegationsService delegationsService) {
        this.delegationsService = delegationsService;
    }

    @Required
    public void setAutoVerifier(AutoVerifier autoVerifier) {
        this.autoVerifier = autoVerifier;
    }

    @Required
    public void setPddAutoVerifier(PDDAutoVerifier pddAutoVerifier) {
        this.pddAutoVerifier = pddAutoVerifier;
    }

    @Required
    public void setHostInfoService(HostInfoService hostInfoService) {
        this.hostInfoService = hostInfoService;
    }

    @Required
    public void setTblUsersHostsDao(TblUsersHostsDao tblUsersHostsDao) {
        this.tblUsersHostsDao = tblUsersHostsDao;
    }

    @Required
    public void setTblUsersHostsCounterDao(TblUsersHostsCounterDao tblUsersHostsCounterDao) {
        this.tblUsersHostsCounterDao = tblUsersHostsCounterDao;
    }

    @Required
    public void setHostEventService(HostEventService hostEventService) {
        this.hostEventService = hostEventService;
    }
}
