package ru.yandex.wmconsole.service;

import java.util.List;

import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.transaction.TransactionStatus;

import ru.yandex.common.framework.user.UserInfo;
import ru.yandex.webmaster.common.WebmasterException;
import ru.yandex.webmaster.common.WebmasterExceptionType;
import ru.yandex.webmaster.common.util.HostnamePart;
import ru.yandex.webmaster.common.xml.limits.LimitDelegationInfo;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.UsersHostsInfo;
import ru.yandex.wmconsole.data.info.limits.HostLimitsInfo;
import ru.yandex.wmconsole.data.info.limits.HostLimitsOrder;
import ru.yandex.wmconsole.data.info.limits.HostsAndLimitsInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmconsole.service.dao.TblHostsMainUsersDao;
import ru.yandex.wmconsole.service.dao.TblXmlLimitDelegationsDao;
import ru.yandex.wmconsole.service.dao.TblXmllimitDelegationsMetaDao;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.service.ViewerUserIdService;
import ru.yandex.wmtools.common.util.ServiceTransactionCallbackWithoutResult;

/**
 * XML Limits Service
 *
 * see http://wiki.yandex-team.ru/JandeksPoisk/Interfejjsy/CentrWebmastera/Specs/XMLandSEO
 *
 * User: azakharov
 * Date: 23.10.12
 * Time: 19:20
 */
public class XmlLimitsService extends AbstractDbService {
    private static final Logger log = LoggerFactory.getLogger(XmlLimitsService.class);

    private static final int BATCH_SIZE = 64;

    private UsersHostsService usersHostsService;
    private ViewerUserIdService viewerUserIdService;
    private TblHostsMainUsersDao tblHostsMainUsersDao;
    private TblXmlLimitDelegationsDao tblXmlLimitDelegationsDao;
    private TblXmllimitDelegationsMetaDao tblXmllimitDelegationsMetaDao;

    public void delegateHostLimit2(final long hostId, final long currentUserId,
            final long delegateToUserId) throws InternalException, UserException
    {
        LimitDelegationInfo delegationInfo = tblXmlLimitDelegationsDao.getDelegationForHost(hostId);
        long delegateFromUserId = 0;
        if (delegationInfo != null) {
            if(delegationInfo.getToUserId() == delegateToUserId) {
                throw new WebmasterException(WebmasterExceptionType.XML_LIMITS__LIMIT_ALREADY_DELEGATED, "Host limit already delegated");
            }
            delegateFromUserId = delegationInfo.getToUserId();
        }
        UsersHostsInfo usersHostsInfo = usersHostsService.getUsersHostsInfo(currentUserId, hostId);
        boolean userIsHostOwner = usersHostsInfo != null && usersHostsInfo.getVerificationState().isVerified();
        if (!userIsHostOwner) {
            UsersHostsInfo recipientUserInfo = usersHostsService.getUsersHostsInfo(delegateToUserId, hostId);
            if (recipientUserInfo == null || !recipientUserInfo.getVerificationState().isVerified()) {
                throw new WebmasterException(WebmasterExceptionType.XML_LIMITS__MUST_DELEGATE_TO_OWNER,
                        "Unable to delegate limit to " + delegateToUserId + " because user isn't host owner");
            }
        }
        viewerUserIdService.assureUserIsInUsersList(delegateToUserId);
        try {
            tblXmlLimitDelegationsDao
                    .insertLimitDelegation(hostId, currentUserId, delegateFromUserId, delegateToUserId);
        } catch (Exception e) {
            throw new WebmasterException(WebmasterExceptionType.INTERNAL__SQL, "Unable to delegate limit", e);
        }
    }

    /**
     * Process main user and limits when host removed or host right rechecked unsuccessfully
     *
     * @param userId        user who removes host
     * @param hostId        host id from user db
     * @throws InternalException
     * @throws UserException
     */
    public void processRemoveHost(final long userId, final long hostId) throws InternalException, UserException {
        getServiceTransactionTemplate(WMCPartition.nullPartition()).executeInService(new ServiceTransactionCallbackWithoutResult() {
            @Override
            protected void doInTransactionWithoutResult(TransactionStatus transactionStatus) throws UserException, InternalException {
                List<UserInfo> verifiedUserForHost = usersHostsService.getVerifiedUserForHost(hostId, false);

                if (verifiedUserForHost.size() > 1) {
                    log.info("There are more host owners than one. Do not clean xml limits delegation");
                    return;
                }

                if (verifiedUserForHost.size() == 1 && verifiedUserForHost.get(0).getUserId() != userId) {
                    log.info("There are more host owners than one. Do not clean xml limits delegation");
                    return;
                }

                log.info("Host is not owned. Remove all xml limit delegations");
                // since there is no host owner, we remove all delegations for the host
                tblXmlLimitDelegationsDao.deleteDelegation(hostId);
            }
        });
    }

    /**
     * Handles host verification event in order to update XML-limit info
     *
     * if user has verified the host, and currently users is host limit owner,
     * it becomes new main host limit owner (it can delegate limit in the future)
     *
     * @param verificationInfo verification info object to be handled
     */
    public void processHostVerified(@NotNull UsersHostsInfo verificationInfo) throws InternalException {
        LimitDelegationInfo delegationInfo = tblXmlLimitDelegationsDao.getDelegationForHost(verificationInfo.getHostId());
        if (delegationInfo != null) {
            log.info("Xml limits already delegated: to_user_id={}", delegationInfo.getToUserId());
        } else {
            log.info("Delegate xml limits to first user: to_user_id={}", verificationInfo.getUserId());
            tblXmlLimitDelegationsDao.insertLimitDelegation(verificationInfo.getHostId(), verificationInfo.getUserId(),
                    verificationInfo.getUserId(), verificationInfo.getUserId());
        }
    }

    public void fixMissingHostLimitDelegations() {
        long lastHostId = 0;
        for (int i = 0; i < 2000; i++) {
            try {
                List<BriefHostInfo> missingLimitDelegations =
                        tblXmllimitDelegationsMetaDao.findMissingLimitDelegations(lastHostId, BATCH_SIZE);
                if (missingLimitDelegations.isEmpty()) {
                    return;
                }
                lastHostId = missingLimitDelegations.get(missingLimitDelegations.size() - 1).getId();
                for (BriefHostInfo briefHostInfo : missingLimitDelegations) {
                    Long mainUserForHost = tblHostsMainUsersDao.getMainUserForHost(briefHostInfo.getId());
                    long delegateToUser;
                    if (mainUserForHost != null) {
                        delegateToUser = mainUserForHost;
                    } else {
                        List<UserInfo> verifiedUserForHost = usersHostsService.getVerifiedUserForHost(briefHostInfo, false);
                        if (verifiedUserForHost.isEmpty()) {
                            log.error("No verified users for host: hostId={}, hostname={}", briefHostInfo.getId(),
                                    briefHostInfo.getName());
                            continue;
                        }
                        delegateToUser = verifiedUserForHost.get(0).getUserId();
                    }
                    log.info("Add missiing xml limit delegation to main user: hostId={}, hostname={}, toUserId={}",
                            briefHostInfo.getId(), briefHostInfo.getName(), delegateToUser);
                    tblXmlLimitDelegationsDao.safeDelegateLimits(briefHostInfo, delegateToUser);
                }
            } catch (InternalException e) {
                log.error("Unable to fix missing xml delagations", e);
            }
        }
    }

    public List<HostLimitsInfo> getLimits(long userId, HostnamePart hostnamePart, HostLimitsOrder order, int pageFrom, int pageSize)
            throws InternalException
    {
        try {
            return tblXmllimitDelegationsMetaDao.getLimits(userId, hostnamePart, order, pageFrom, pageSize);
        } catch (Exception e) {
            throw new WebmasterException(WebmasterExceptionType.INTERNAL__SQL, "Unable to get host limits", e);
        }
    }

    public HostsAndLimitsInfo getHostsAndLimits(long userId, HostnamePart hostnameFilter) throws InternalException {
        try {
            return tblXmllimitDelegationsMetaDao.getTotalLimit(userId, hostnameFilter);
        } catch (Exception e) {
            throw new WebmasterException(WebmasterExceptionType.INTERNAL__SQL, "Unable to get aggregate limits", e);
        }
    }

    @Required
    public void setViewerUserIdService(ViewerUserIdService viewerUserIdService) {
        this.viewerUserIdService = viewerUserIdService;
    }

    @Required
    public void setTblHostsMainUsersDao(TblHostsMainUsersDao tblHostsMainUsersDao) {
        this.tblHostsMainUsersDao = tblHostsMainUsersDao;
    }

    @Required
    public void setTblXmlLimitDelegationsDao(TblXmlLimitDelegationsDao tblXmlLimitDelegationsDao) {
        this.tblXmlLimitDelegationsDao = tblXmlLimitDelegationsDao;
    }

    @Required
    public void setTblXmllimitDelegationsMetaDao(TblXmllimitDelegationsMetaDao tblXmllimitDelegationsMetaDao) {
        this.tblXmllimitDelegationsMetaDao = tblXmllimitDelegationsMetaDao;
    }

    @Required
    public void setUsersHostsService(UsersHostsService usersHostsService) {
        this.usersHostsService = usersHostsService;
    }
}
