package ru.yandex.wmconsole.service;

import java.io.IOException;
import java.net.URL;
import java.util.Collection;
import java.util.Collections;

import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElementWrapper;
import javax.xml.bind.annotation.XmlRootElement;

import org.apache.http.HttpHost;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.wmconsole.service.error.WMCUserProblem;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.error.UserProblem;
import ru.yandex.wmtools.common.servantlet.AbstractServantlet;
import ru.yandex.wmtools.common.util.HttpConnector;
import ru.yandex.wmtools.common.util.HttpResponse;

/**
 * User: azakharov
 * Date: 22.05.12
 * Time: 16:44
 */
public class ZoraService {
    private static final Logger log = LoggerFactory.getLogger(ZoraService.class.getName());

    // Заголовок в запросе, идентифицирующий источник запроса
    private static final String SOURCE_NAME_HEADER_NAME = "X-Yandex-Sourcename";
    private static final String SOURCE_NAME_HEADER_VALUE = "WebmasterRenewer";
    // Заголовок в ответе, поясняющий причину ошибки или информацию об успешном обновлении
    private static final String STATUS_HEADER_NAME = "X-Yandex-Status";

    private URL zoraUrl;
    private int connectionTimeoutMillis = 5000;
    private int socketTimeoutMillis = 5000;

    /**
     * Делает http-запрос к ZORA для обновления информации по хосту.
     *
     * @param host
     */
    public ZoraResponse renewHost(String host) throws UserException {
        URL url = AbstractServantlet.prepareUrl(host, true);

        try {
            HttpResponse response = new HttpConnector.RequestBuilder(url)
                    .setProxy(new HttpHost(zoraUrl.getHost(), zoraUrl.getPort()))
                    .connectionTimeout(connectionTimeoutMillis)
                    .socketTimeout(socketTimeoutMillis)
                    .header(SOURCE_NAME_HEADER_NAME, SOURCE_NAME_HEADER_VALUE)
                    .execute();

            log.debug("Zora response for " + host + ": " + response.getStatusCode());
            Collection<String> values = response.getHeaders().get(STATUS_HEADER_NAME);
            if (values != null) {
                for (String value : values) {
                    log.debug(STATUS_HEADER_NAME + ": " + value);
                }
            }

            if (response.getStatusCode() != 202) {
                throw new UserException(
                        response.getStatusCode() == 500 ? UserProblem.ZORA_REQUEST_TOO_FREQUENT : UserProblem.ZORA_UNSPECIFIED_ERROR,
                        "Error http code " + response.getStatusCode() + " from " + zoraUrl);
            }

            return new ZoraResponse(response.getStatusCode(),
                    values != null ? values : Collections. <String>emptyList());
        } catch (IOException e) {
            log.error("IOException in http connection to zora", e);
            throw new UserException(WMCUserProblem.URL_CONNECTION_PROBLEM, "zora service connection problem");
        }
    }

    @XmlRootElement(name = "response")
    public static class ZoraResponse {
        private int httpStatus;
        private Collection<String> yandexStatus;

        public ZoraResponse() {
            // for JAXB
        }

        public ZoraResponse(int httpStatus, Collection<String> yandexStatus) {
            this.httpStatus = httpStatus;
            this.yandexStatus = yandexStatus;
        }

        @XmlElement(name = "http-code")
        public int getHttpStatus() {
            return httpStatus;
        }

        @XmlElementWrapper(name="headers")
        @XmlElement(name="x-yandex-status")
        public Collection<String> getYandexStatus() {
            return yandexStatus;
        }
    }

    @Required
    public void setZoraUrl(URL zoraUrl) {
        this.zoraUrl = zoraUrl;
    }

    @Required
    public void setConnectionTimeoutMillis(int connectionTimeoutMillis) {
        this.connectionTimeoutMillis = connectionTimeoutMillis;
    }

    @Required
    public void setSocketTimeoutMillis(int socketTimeoutMillis) {
        this.socketTimeoutMillis = socketTimeoutMillis;
    }
}
