package ru.yandex.wmconsole.service.dao;

import org.jetbrains.annotations.Nullable;
import org.springframework.jdbc.core.ColumnMapRowMapper;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import ru.yandex.common.util.collections.Cu;
import ru.yandex.common.util.db.LongRowMapper;
import ru.yandex.wmconsole.data.info.HostDbHostInfo;
import ru.yandex.wmconsole.data.info.all.about.url.ShortUrlInfo;
import ru.yandex.wmconsole.data.info.all.about.url.UrlRequestId;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.util.SqlUtil;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;

/**
 * User: azakharov
 * Date: 24.10.13
 * Time: 14:44
 */
public class TblAllAboutUrlQueueDao extends AbstractDbService {

    public void insertRequest(final HostDbHostInfo hostDbHostInfo, String url, boolean fromSupport) throws InternalException {
        final String query = "INSERT INTO tbl_all_about_url_queue (host_id, create_date, url, processed, from_support) VALUES (?, NOW(), ?, 0, ?)";
        getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).update(
                query, hostDbHostInfo.getHostDbHostId(), url.toString(), fromSupport);
    }

    public int getUrlInfoRequestCount(final HostDbHostInfo hostDbHostInfo, final boolean fromSupport) throws InternalException {
        final String query = "SELECT COUNT(*) FROM tbl_all_about_url_queue WHERE host_id = ? AND processed = 0 AND from_support = ? ";
        return getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).queryForInt(
                query, hostDbHostInfo.getHostDbHostId(), fromSupport);
    }

    public List<ShortUrlInfo> getUrlInfoQueue(final HostDbHostInfo hostDbHostInfo, final boolean fromSupport) throws InternalException {
        final String SELECT_URL_QUEUE_QUERY =
                "SELECT num, url, create_date " +
                        "FROM " +
                        "   tbl_all_about_url_queue " +
                        "WHERE " +
                        "   host_id = ? " +
                        "AND " +
                        "   from_support = ? " +
                        "AND " +
                        "   processed = 0 " +
                        "ORDER BY create_date DESC";

        return getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).query(
                SELECT_URL_QUEUE_QUERY, queueListMapper, hostDbHostInfo.getHostDbHostId(), fromSupport);
    }

    public List<ShortUrlInfo> getUrlInfoQueue(final HostDbHostInfo hostDbHostInfo,
                                              final List<Long> ids,
                                              final boolean fromSupport) throws InternalException {
        final String SELECT_URL_QUEUE_QUERY =
                "SELECT num, url, create_date " +
                        "FROM " +
                        "   tbl_all_about_url_queue " +
                        "WHERE " +
                        "   host_id = ? " +
                        "AND " +
                        "   from_support = ? " +
                        "AND " +
                        "   processed = 0 " +
                        "AND " +
                        "   num IN (" + SqlUtil.getCommaSeparatedList(ids)+ ") " +
                        "ORDER BY create_date DESC";

        return getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).query(
                SELECT_URL_QUEUE_QUERY, queueListMapper, hostDbHostInfo.getHostDbHostId(), fromSupport);
    }

    public List<ShortUrlInfo> getUrlInfoQueueForUrl(final HostDbHostInfo hostDbHostInfo, final String url, final boolean fromSupport) throws InternalException {
        final String query =
                "SELECT num, url, create_date " +
                        "FROM " +
                        "   tbl_all_about_url_queue " +
                        "WHERE " +
                        "   host_id = ? " +
                        "AND " +
                        "   from_support = ? " +
                        "AND " +
                        "   url = ? " +
                        "AND " +
                        "   processed = 0 " +
                        "ORDER BY create_date DESC";

        return getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).query(
                query, queueListMapper, hostDbHostInfo.getHostDbHostId(), fromSupport, url);
    }

    public List<Long> getOldestUrlInfoRequests(final HostDbHostInfo hostDbHostInfo, int limit, boolean fromSupport) throws InternalException {
        final String SELECT_OLDEST_IN_QUEUE_QUERY =
                "SELECT num FROM tbl_all_about_url_queue " +
                        "WHERE host_id = ? AND processed = 0 AND from_support = ? " +
                        "ORDER BY create_date ASC LIMIT %d";

        return getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).query(
                String.format(SELECT_OLDEST_IN_QUEUE_QUERY, limit),
                new LongRowMapper(),
                hostDbHostInfo.getHostDbHostId(), fromSupport);
    }

    public @Nullable
    ShortUrlInfo getUrlInfoQueueSummary(HostDbHostInfo hostDbHostInfo, UrlRequestId requestId, boolean fromSupport) throws InternalException {
        final String query =
                "SELECT num, url, create_date " +
                        "FROM " +
                        "   tbl_all_about_url_queue " +
                        "WHERE " +
                        "   host_id = ? " +
                        "AND " +
                        "   from_support = ? " +
                        "AND " +
                        "   num = ? " +
                        "AND " +
                        "   processed = 0";
        return getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).safeQueryForObject(
                query, queueListMapper, hostDbHostInfo.getHostDbHostId(), fromSupport, requestId.getId());
    }

    public void deleteUrlInfoRequests(final HostDbHostInfo hostDbHostInfo, final List<Long> requestIds) throws InternalException {
        final String query = "UPDATE tbl_all_about_url_queue SET processed = 1 WHERE host_id = ? AND num IN (%s)";
        getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).update(
                String.format(query, SqlUtil.getCommaSeparatedList(requestIds)),
                hostDbHostInfo.getHostDbHostId()
        );
    }

    public void deleteUrlInfoRequest(final HostDbHostInfo hostDbHostInfo, final UrlRequestId requestId, final boolean fromSupport) throws InternalException {
        final String query = "UPDATE tbl_all_about_url_queue SET processed = 1 WHERE host_id = ? AND num = ? AND from_support = ? ";
        getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).update(
                query, hostDbHostInfo.getHostDbHostId(), requestId.getId(), fromSupport);
    }

    public Map<String, Object> dumpAllAboutUrlQueueInfo(final HostDbHostInfo hostDbHostInfo, final UrlRequestId id) throws InternalException {
        final String q = "SELECT num, host_id, create_date, url, processed, from_support, retry_date " +
                             "FROM tbl_all_about_url_queue WHERE num = ? ";
        List<Map<String, Object>> rs = getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).query(q, dumpTableMapper, id.getId());
        return Cu.emptyIfNull(Cu.firstOrNull(rs));
    }

    private static final ParameterizedRowMapper<Map<String,Object>> dumpTableMapper = new MapMapper();
    public static class MapMapper extends ColumnMapRowMapper implements ParameterizedRowMapper<Map<String, Object>> {
    }

    private static final ParameterizedRowMapper<ShortUrlInfo> queueListMapper = new ParameterizedRowMapper<ShortUrlInfo>() {
        private static final String FIELD_NUM = "num";
        private static final String FIELD_URL = "url";
        private static final String FIELD_CREATE_DATE = "create_date";

        @Override
        public ShortUrlInfo mapRow(ResultSet rs, int rowNum) throws SQLException {
            return ShortUrlInfo.createUrlInfoFromQueue(
                    new UrlRequestId(rs.getLong(FIELD_NUM)),
                    rs.getString(FIELD_URL),
                    rs.getTimestamp(FIELD_CREATE_DATE));
        }
    };
}
