package ru.yandex.wmconsole.service.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.List;

import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;

import ru.yandex.common.framework.pager.Pager;
import ru.yandex.wmconsole.data.BlockedUserInfo;
import ru.yandex.wmconsole.data.UserBlockReason;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.util.IServiceJdbcTemplate;

/**
 * @author aherman
 */
public class TblBlockedUsersDao extends AbstractDbService {
    public List<BlockedUserInfo> getBlockedUsersList(Pager pager) throws InternalException {
        String countQ = "SELECT count(*) FROM tbl_blocked_users ";
        String q = "SELECT * FROM tbl_blocked_users ORDER BY user_id %1$s ";

        return getJdbcTemplate(WMCPartition.nullPartition()).pageableSelect(countQ, q, blockedUserMapper(), pager);
    }

    @Nullable
    public BlockedUserInfo getBlockInfo(long userId) throws InternalException {
        String q = "SELECT * FROM tbl_blocked_users WHERE user_id = ?";
        List<BlockedUserInfo> blockInfoList = getJdbcTemplate(WMCPartition.nullPartition()).query(q, blockedUserMapper(),
                userId);
        if (blockInfoList.isEmpty()) {
            return null;
        }
        return blockInfoList.get(0);
    }

    public boolean isUserBlocked(long userId) throws InternalException {
        String q = "SELECT count(*) FROM tbl_blocked_users WHERE user_id = ?";
        long count = getJdbcTemplate(new WMCPartition(null, userId)).queryForLong(q, userId);
        return (count != 0);
    }

    public void addBlockedUser(IServiceJdbcTemplate jdbcTemplate, long userId, UserBlockReason blockReason,
            long blockedByUserId) throws InternalException
    {
        String q = "INSERT INTO tbl_blocked_users (user_id, block_reason, blocked_by_user_id) VALUES (?, ?, ?)";
        if (jdbcTemplate.update(q, userId, blockReason.getValue(), blockedByUserId) == 0) {
            throw new RuntimeException("No rows updated while adding blocked user into DB.");
        }
    }

    public int removeBlockedUser(IServiceJdbcTemplate jdbcTemplate, Long... userId) throws InternalException {
        String inParamsStr = "";
        final Object[] queryParams = new Object[userId.length];
        for (int i = 0; i < userId.length; i++) {
            if (i != 0) {
                inParamsStr += ", ";
            }
            inParamsStr += "?";
            queryParams[i] = userId[i];
        }
        final String q = String.format("DELETE FROM tbl_blocked_users WHERE user_id IN (%1$s)", inParamsStr);
        return jdbcTemplate.update(q, queryParams);
    }

    private static ParameterizedRowMapper<BlockedUserInfo> blockedUserMapper() {
        return new ParameterizedRowMapper<BlockedUserInfo>() {
            @Override
            public BlockedUserInfo mapRow(ResultSet rs, int rowNum) throws SQLException {
                long userId = rs.getLong("user_id");
                long blockedByUserId = rs.getLong("blocked_by_user_id");
                int blockReasonId = rs.getInt("block_reason");
                Timestamp blockDate = rs.getTimestamp("block_date");
                return new BlockedUserInfo(userId, blockedByUserId, UserBlockReason.R.fromValueOrUnknown(blockReasonId),
                        new DateTime(blockDate.getTime()));
            }
        };
    }
}
