package ru.yandex.wmconsole.service.dao;

import org.apache.commons.lang.StringUtils;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import ru.yandex.common.framework.pager.Pager;
import ru.yandex.common.util.collections.Cu;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.mirror.MainMirrorHistoryInfo;
import ru.yandex.wmconsole.data.mirror.MirrorGroupActionEnum;
import ru.yandex.wmconsole.data.mirror.MirrorGroupChangeRequest;
import ru.yandex.wmconsole.data.mirror.MirrorGroupChangeStateEnum;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.sita.SitaMirroringActionStatusEnum;
import ru.yandex.wmtools.common.sita.SitaMirroringHostStatusEnum;
import ru.yandex.wmtools.common.util.SqlUtil;
import ru.yandex.wmtools.common.util.TimeFilter;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * User: azakharov
 * Date: 24.03.14
 * Time: 16:30
 */
public class TblMainMirrorHistoryDao extends AbstractDbService {

    public void saveRequest(MirrorGroupChangeRequest request) throws InternalException {

        final String query =
                "INSERT INTO tbl_main_mirror_history " +
                        "   (host_id, state, action_id, desired_main, created_at, modified_at, user_id, old_main) " +
                        "VALUES " +
                        "   (?, ?, ?, ?, ?, ?, ?, ?)";

        getJdbcTemplate(WMCPartition.nullPartition()).insertSingle(query,
                request.getHostId(),
                request.getState().value(),
                request.getAction().value(),
                request.getDesiredMain(),
                request.getCreateDate(),
                request.getModificationDate(),
                request.getUserId(),
                request.getOldMainMirrorName());
    }

    public List<MainMirrorHistoryInfo> getHistory(Pager pager,
                           TimeFilter creationDateFilter,
                           TimeFilter modifyingDateFilter,
                           String hostNameFilter,
                           MirrorGroupActionEnum actionFilter,
                           MirrorGroupChangeStateEnum requestStateFilter) throws InternalException {

        List<Object> params = new ArrayList<Object>();
        SqlCondition hostNameLikeCondition = hostNameLikeCondition(hostNameFilter);
        params.addAll(hostNameLikeCondition.args());
        SqlCondition actionCondition = actionCondition(actionFilter);
        params.addAll(actionCondition.args());
        SqlCondition stateCondition = stateCondition(requestStateFilter);
        params.addAll(stateCondition.args());
        SqlCondition createdAtCondition = timeFilterCondition(creationDateFilter, "created_at");
        params.addAll(createdAtCondition.args());
        SqlCondition modifiedAtCondition = timeFilterCondition(modifyingDateFilter, "modified_at");
        params.addAll(modifiedAtCondition.args());

        SqlCondition allFilters = SqlCondition.all(
                hostNameLikeCondition,
                actionCondition,
                stateCondition,
                createdAtCondition,
                modifiedAtCondition
        );

        String query =
                "SELECT user_id, mh.host_id, h.name, state, action_id, created_at, modified_at, desired_main, " +
                        "action_status, host1, status1, host2, status2, old_main " +
                "FROM tbl_main_mirror_history mh JOIN tbl_hosts h ON (mh.host_id = h.host_id) " +
                "WHERE " + allFilters.sql() + " " +
                "ORDER BY created_at DESC %1$s ";

        String countQuery =
                "SELECT COUNT(*) " +
                "FROM tbl_main_mirror_history mh JOIN tbl_hosts h ON (mh.host_id = h.host_id) " +
                "WHERE " + allFilters.sql() + " " +
                "ORDER BY created_at DESC ";

        return getJdbcTemplate(WMCPartition.nullPartition()).pageableSelect(countQuery, query, m, pager, params.toArray());
    }

    public MainMirrorHistoryInfo getLastSuccessfulRequest(long hostId) throws InternalException {
        String query =
                "SELECT user_id, mh.host_id, h.name, state, action_id, created_at, modified_at, desired_main, " +
                        "action_status, host1, status1, host2, status2, old_main " +
                        "FROM tbl_main_mirror_history mh JOIN tbl_hosts h ON (mh.host_id = h.host_id) " +
                        "WHERE mh.host_id = ? AND action_id = " + MirrorGroupActionEnum.RERANGE.value() + " " +
                        "AND state IN (" + MirrorGroupChangeStateEnum.CHECKED.value() + ", " + MirrorGroupChangeStateEnum.ACCEPTED.value() + ") " +
                        "ORDER BY created_at, modified_at DESC LIMIT 1";
        List<MainMirrorHistoryInfo> res = getJdbcTemplate(WMCPartition.nullPartition()).query(query, m, hostId);

        return (res.isEmpty()) ? null : Cu.first(res);
    }

    public boolean hasFailedRequestsAfter(MainMirrorHistoryInfo r) throws InternalException {
        String query =
                "SELECT COUNT(*) FROM tbl_main_mirror_history "+
                        "WHERE " +
                        "    host_id = ? " +
                        "AND action_id = ? " +
                        "AND desired_main = ? "+
                        "AND created_at > ? " +
                        "AND state IN (" + MirrorGroupChangeStateEnum.DECLINED.value() + ", " + MirrorGroupChangeStateEnum.RECHECK_DECLINED.value() + ") ";
        long count = getJdbcTemplate(WMCPartition.nullPartition()).queryForLong(query, r.getHostId(), r.getAction().value(), r.getDesiredMain(), r.getCreatedAt());
        return count > 0;
    }

    private static SqlCondition hostNameLikeCondition(String hostNamePart) {
        if (StringUtils.isEmpty(hostNamePart)) {
            return SqlCondition.trueCondition();
        }
        return SqlCondition.column("h.name").like("%" + hostNamePart + "%");
    }

    private static SqlCondition actionCondition(MirrorGroupActionEnum action) {
        if (action == null) {
            return SqlCondition.trueCondition();
        }
        return SqlCondition.column("action_id").eq(action.value());
    }

    private static SqlCondition stateCondition(MirrorGroupChangeStateEnum requestStateFilter) {
        if (requestStateFilter == null) {
            return SqlCondition.trueCondition();
        }
        return SqlCondition.column("state").eq(requestStateFilter.value());
    }

    private static SqlCondition timeFilterCondition(final TimeFilter createAtTimeFilter, final String fieldName) {
        final SqlCondition fromCondition;
        if (createAtTimeFilter.getFromInMilliseconds() != null) {
            fromCondition = SqlCondition.column(fieldName).ge(new Date(createAtTimeFilter.getFromInMilliseconds()));
        } else {
            fromCondition = SqlCondition.trueCondition();
        }

        final SqlCondition toCondition;
        if (createAtTimeFilter.getToInMilliseconds() != null) {
            toCondition = SqlCondition.column(fieldName).le(new Date(createAtTimeFilter.getToInMilliseconds()));
        } else {
            toCondition = SqlCondition.trueCondition();
        }

        return SqlCondition.all(fromCondition, toCondition);
    }

    private ParameterizedRowMapper<MainMirrorHistoryInfo> m = new ParameterizedRowMapper<MainMirrorHistoryInfo>() {
        @Override
        public MainMirrorHistoryInfo mapRow(ResultSet rs, int rowNum) throws SQLException {
            int actionStatusValue = rs.getInt("action_status");
            final SitaMirroringActionStatusEnum actionStatus =
                    rs.wasNull() ? null : SitaMirroringActionStatusEnum.R.fromValueOrNull(actionStatusValue);
            final String hostName1 = SqlUtil.getStringNullable(rs, "host1");
            int val1 = rs.getInt("status1");
            final SitaMirroringHostStatusEnum hostStatus1 =
                    rs.wasNull() ? null : SitaMirroringHostStatusEnum.R.fromValueOrNull(val1);
            final String hostName2 = SqlUtil.getStringNullable(rs, "host2");
            int val2 = rs.getInt("status2");
            final SitaMirroringHostStatusEnum hostStatus2 =
                    rs.wasNull() ? null : SitaMirroringHostStatusEnum.R.fromValueOrNull(val2);

            return new MainMirrorHistoryInfo(
                    rs.getLong("user_id"),
                    rs.getString("name"),
                    SqlUtil.safeGetTimestamp(rs, "modified_at"),
                    SqlUtil.safeGetTimestamp(rs, "created_at"),
                    SqlUtil.getStringNullable(rs, "desired_main"),
                    MirrorGroupActionEnum.R.fromValueOrNull(rs.getInt("action_id")),
                    MirrorGroupChangeStateEnum.R.fromValueOrNull(rs.getInt("state")),
                    rs.getLong("host_id"),
                    actionStatus,
                    hostName1,
                    hostStatus1,
                    hostName2,
                    hostStatus2,
                    SqlUtil.getStringNullable(rs, "old_main")
            );
        }
    };
}
