package ru.yandex.wmconsole.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import ru.yandex.common.util.StringUtils;
import ru.yandex.common.util.collections.CollectionFactory;

/**
 * User: azakharov
 * Date: 02.11.12
 * Time: 17:54
 */
public class SeoRequestParser extends DefaultHandler {
    private static final String TAG_REQUEST_REPORT = "request-report";
    private static final String TAG_QUERIES_LIST = "queries-list";
    private static final String TAG_QUERY = "query";
    private static final String TAG_TEXT = "text";
    private static final String TAG_REGIONS = "regions";
    private static final String TAG_REGION_ID = "rid";
    private static final String TAG_TOP_SIZE = "top-size";
    private static final String TAG_DEFAULT_TOP_SIZE = "default-top-size";
    private static final String TAG_DEFAULT_REGION = "default-region";

    private List<Query> queries = new LinkedList<Query>();
    private Long defaultTopSize;
    private Long defaultRegion;
    private List<String> parseStack = new ArrayList<String>(5);
    private Query currentQuery = null;
    private Map<String, List<String>> tagToExpectedStack = new HashMap<String, List<String>>();

    public SeoRequestParser() {
        tagToExpectedStack.put(TAG_REQUEST_REPORT,
                new ArrayList<String>(CollectionFactory.list(TAG_REQUEST_REPORT)));
        tagToExpectedStack.put(TAG_QUERIES_LIST,
                new ArrayList<String>(CollectionFactory.list(TAG_REQUEST_REPORT, TAG_QUERIES_LIST)));
        tagToExpectedStack.put(TAG_QUERY,
                new ArrayList<String>( CollectionFactory.list(TAG_REQUEST_REPORT, TAG_QUERIES_LIST, TAG_QUERY)));
        tagToExpectedStack.put(TAG_TEXT,
                new ArrayList<String>(CollectionFactory.list(TAG_REQUEST_REPORT, TAG_QUERIES_LIST, TAG_QUERY, TAG_TEXT)));
        tagToExpectedStack.put(TAG_REGIONS,
                new ArrayList<String>(CollectionFactory.list(TAG_REQUEST_REPORT, TAG_QUERIES_LIST, TAG_QUERY, TAG_REGIONS)));
        tagToExpectedStack.put(TAG_REGION_ID,
                new ArrayList<String>(CollectionFactory.list(TAG_REQUEST_REPORT, TAG_QUERIES_LIST, TAG_QUERY, TAG_REGIONS, TAG_REGION_ID)));
        tagToExpectedStack.put(TAG_TOP_SIZE,
                new ArrayList<String>(CollectionFactory.list(TAG_REQUEST_REPORT, TAG_QUERIES_LIST, TAG_QUERY, TAG_TOP_SIZE)));
        tagToExpectedStack.put(TAG_DEFAULT_TOP_SIZE,
                new ArrayList<String>(CollectionFactory.list(TAG_REQUEST_REPORT, TAG_QUERIES_LIST, TAG_DEFAULT_TOP_SIZE)));
        tagToExpectedStack.put(TAG_DEFAULT_REGION,
                new ArrayList<String>(CollectionFactory.list(TAG_REQUEST_REPORT, TAG_QUERIES_LIST, TAG_DEFAULT_REGION)));
    }

    private void checkStack(List<String> expectedStack) throws SAXException {
        for (int i = 0; i < expectedStack.size(); i++) {
            if (parseStack.size() <= i ||
                    parseStack.get(i) != expectedStack.get(i)) {
                throw new SAXException("Invalid parse stack. Expected  " +
                        StringUtils.join(expectedStack, ":") + " found " +
                        StringUtils.join(parseStack, ":"));
            }
        }
    }

    @Override
    public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {
        parseStack.add(qName);
        elementValue.setLength(0);
        List<String> expectedStack = tagToExpectedStack.get(qName);
        if (expectedStack == null) {
            throw new SAXException("unknown tag " + qName);
        } else {
            checkStack(expectedStack);
        }
        if (TAG_QUERY.equals(qName)) {
            currentQuery = new Query();
            queries.add(currentQuery);
        }
    }

    StringBuilder elementValue = new StringBuilder();
    @Override
    public void characters(char[] ch, int start, int length) throws SAXException {
        elementValue.append(ch, start, length);
    }

    @Override
    public void endElement(String uri, String localName, String qName) throws SAXException {
        if (!parseStack.isEmpty()) {
            parseStack.remove(parseStack.size() - 1);
        }
        if (TAG_TEXT.equals(qName)) {
            currentQuery.setQuery(elementValue.toString());
        } else if (TAG_REGION_ID.equals(qName)) {
            Long regionId = this.parseLong(elementValue.toString());
            currentQuery.getRegions().add(regionId);
        } else if (TAG_TOP_SIZE.equals(qName)) {
            Long topSize = this.parseLong(elementValue.toString());
            currentQuery.setTopSize(topSize);
        } else if (TAG_DEFAULT_TOP_SIZE.equals(qName)) {
            if (defaultTopSize != null) {
                throw new SAXException("duplicate value of " + TAG_DEFAULT_TOP_SIZE);
            }
            defaultTopSize = this.parseLong(elementValue.toString());
        } else if (TAG_DEFAULT_REGION.equals(qName)) {
            if (defaultRegion != null) {
                throw new SAXException("duplicate value of " + TAG_DEFAULT_REGION);
            }
            defaultRegion = this.parseLong(elementValue.toString());
        } else if (TAG_QUERY.equals(qName)) {
            if (currentQuery.getQuery() == null ||
                    currentQuery.getQuery() != null &&
                    currentQuery.getQuery().trim().isEmpty()) {
                throw new SAXException("no query text or query text is empty");
            }
        }
    }

    private Long parseLong(String value) throws SAXException {
        try {
            return Long.valueOf(value);
        } catch (NumberFormatException nfe) {
            throw new SAXException("unable to parse value " + value);
        }
    }

    public List<Query> getQueries() {
        return queries;
    }

    public Long getDefaultTopSize() {
        return defaultTopSize;
    }

    public Long getDefaultRegion() {
        return defaultRegion;
    }

    public static class Query {
        private String query;
        private List<Long> regions = new LinkedList<Long>();
        private Long topSize;

        public Query() {
        }

        public String getQuery() {
            return query;
        }

        public List<Long> getRegions() {
            return regions;
        }

        public Long getTopSize() {
            return topSize;
        }

        public void setQuery(String query) {
            this.query = query;
        }

        public void setRegions(List<Long> regions) {
            this.regions = regions;
        }

        public void setTopSize(Long topSize) {
            this.topSize = topSize;
        }
    }
}
