package ru.yandex.wmconsole.verification;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * @author baton, amima
 */
public class CheckMetaTagContentHandler extends DefaultHandler {

    private static final Logger log = LoggerFactory.getLogger(CheckMetaTagContentHandler.class);

    // is tag with right 'name' and 'content' found?
    private boolean found = false;

    private static final int FAULT_LOG_LEVELS = 4;

    // what was an error?
    private final String[] faultLog = new String[FAULT_LOG_LEVELS];

    private final String name;
    private final String content;
    private final boolean strict;

    private boolean inHead = false;
    private boolean bodyStarted = false;

    public CheckMetaTagContentHandler(String name, String content) {
        this(name, content, true);
    }

    public CheckMetaTagContentHandler(String name, String content, boolean strict) {
        this.name = name;
        this.content = content;
        this.strict = strict;
    }

    public boolean isFound() {
        return found;
    }

    public String getFaultLog() {
        for (int i = FAULT_LOG_LEVELS - 1; i >= 0; i--) {
            if (faultLog[i] != null) {
                return faultLog[i];
            }
        }

        return "";
    }

    @Override
    public void startElement(String uri, String localName, String qName, Attributes atts) throws SAXException {
        if (!"meta".equalsIgnoreCase(localName)) {
            faultLog[0] = "SAX. Checking meta-tag: No tag 'meta' found";
            if ("head".equalsIgnoreCase(localName)) {
                inHead = true;
            } else if (!inHead && "body".equalsIgnoreCase(localName)) {
                bodyStarted = true;
            }

            return;
        }
        log.debug("SAX. Attributes for found tag: name = '" + atts.getValue("name") + "', content = '" + atts.getValue("content") + "' and " + atts.getLength() + " attributes.");

        if (atts.getLength() < 2) {
            faultLog[1] = "SAX. Checking meta-tag: Illegal attributes list length (expected 2 or more, found " + atts.getLength() + ")";
            log.debug(faultLog[1]);
            return;
        }

        if (!name.equalsIgnoreCase(atts.getValue("name"))) {
            faultLog[2] = "SAX. Checking meta-tag: Illegal content or absence of attribute 'name' (expected '" + name + "', found '" + atts.getValue("name") + "')";
            log.debug(faultLog[2]);
            return;
        }

        if ((strict && !content.equalsIgnoreCase(atts.getValue("content"))) ||
                (!strict && atts.getValue("content").toLowerCase().indexOf(content.toLowerCase()) == -1)) {
            faultLog[3] = "SAX. Checking meta-tag: Illegal content or absence of attribute 'content' (expected '" + content + "', found '" + atts.getValue("content") + "')";
            log.debug(faultLog[3]);
            return;
        }

        if (!inHead || bodyStarted) {
            faultLog[3] = "SAX. Checking meta-tag: Meta tag must be in 'head', 'content' = " + atts.getValue("content");
            log.debug(faultLog[3]);
            return;
        }

        found = true;
        log.info("SAX. Checking meta-tag: Verification info in meta tag found successfully.");
    }

    @Override
    public void endElement(String uri, String localName, String qName) throws SAXException {
        if ("head".equalsIgnoreCase(localName)) {
            inHead = false;
        }
    }
}
