package ru.yandex.wmconsole.verification;

import java.security.Security;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.xbill.DNS.Cache;
import org.xbill.DNS.Lookup;
import org.xbill.DNS.Record;
import org.xbill.DNS.TextParseException;
import org.xbill.DNS.Type;

import ru.yandex.wmconsole.data.VerificationStateEnum;
import ru.yandex.wmconsole.data.info.UsersHostsInfo;
import ru.yandex.wmconsole.service.DNSVerificationsService;
import ru.yandex.wmtools.common.SupportedProtocols;
import ru.yandex.wmtools.common.error.InternalException;

/**
 * @author baton
 */
public class DNSVerifier implements Verifier {
    private static final Logger log = LoggerFactory.getLogger(DNSVerifier.class);
    private static final String YANDEX_VERIFICATION_STRING = "yandex-verification";

    private DNSVerificationsService dnsVerificationsService;

    private int dnsCacheTtl;
    private int dnsNegativeCacheTtl;

    public void init() {
        Security.setProperty("networkaddress.cache.ttl", String.valueOf(dnsCacheTtl));
        System.setProperty("sun.net.inetaddr.ttl", String.valueOf(dnsCacheTtl));
        Security.setProperty("networkaddress.cache.negative.ttl", String.valueOf(dnsNegativeCacheTtl));
        System.setProperty("sun.net.inetaddr.negative.ttl", String.valueOf(dnsNegativeCacheTtl));
    }

    @Override
    public UsersHostsInfo verify(UsersHostsInfo verInfo) {
        UsersHostsInfo newInfo = checkVerification(verInfo);
        try {
            newInfo = dnsVerificationsService.handleDNSVerificationResult(newInfo,
                    !VerificationStateEnum.LONG_DNS_WAITING.equals(verInfo.getVerificationState()));
        } catch (InternalException e) {
            log.warn("Failed to handle long dns verification for user " +
                    verInfo.getUserInfo().getLogin() + " and host " + verInfo.getHostName() + "!", e);
        }
        return newInfo;
    }

    @Override
    public UsersHostsInfo cancel(UsersHostsInfo verInfo) {
        return checkVerification(verInfo);
    }

    public UsersHostsInfo checkVerification(UsersHostsInfo verInfo) {
        log.debug("Verifying DNS at " + verInfo.getHostName() + "...");
        try {
            return checkDns(verInfo);
        } catch (TextParseException e) {
            String verifyFaultLog = "Not verified: TextParseException while reading DNS information.\n" + e;
            log.debug(verifyFaultLog);
            return UsersHostsInfo.createNotVerifiedNow(verInfo, VerificationStateEnum.DNS_TEXT_PARSE_EXCEPTION, verifyFaultLog);
        }
    }

    private UsersHostsInfo checkDns(UsersHostsInfo verInfo) throws TextParseException {
        String content = getDnsVerificationRecordContent(verInfo.getVerificationUin());

        // analyzing DNS records
        String hostName = null;
        try {
            hostName = SupportedProtocols.getURL(verInfo.getHostName()).getHost().toLowerCase();
        } catch (Exception e) {
            //ignore
        }
        log.debug("DNS requesting: " + hostName);
        Lookup lookup = new Lookup(hostName, Type.TXT);
        Cache dnsCache = new Cache();
        dnsCache.setMaxCache(dnsCacheTtl);
        dnsCache.setMaxNCache(dnsNegativeCacheTtl);
        lookup.setCache(dnsCache);

        Record[] records = lookup.run();
        log.debug("DNS lookup result: " + lookup.getResult());
        log.debug("DNS lookup error string: " + lookup.getErrorString());
        if (records != null) {
            for (Record record : records) {
                log.debug("DNS record: " + record);
                String str = record.rdataToString();
                log.debug("DNS record rdata: " + str);
                if (record.getType() == Type.TXT) {
                    log.debug("DNS: required content: " + content);
                    log.debug("DNS: found content   : " + str);
                    if (str.contains(YANDEX_VERIFICATION_STRING) && str.contains(content)) {
                        log.debug("Verified successfully - needed DNS TXT record found.");
                        return UsersHostsInfo.createVerifiedNow(verInfo);
                    } else {
                        log.debug("DNS record content differs from expected");
                    }
                } else {
                    log.debug("DNS type is not TXT");
                }
            }
        } else {
            log.debug("DNS lookup returned NULL");
        }

        // record not found!
        return UsersHostsInfo.createNotVerifiedNow(verInfo, VerificationStateEnum.DNS_NO_SUCH_ENTRY, "");
    }

    private String getDnsVerificationRecordContent(long verificationUin) {
        return VerificationIdentifier.valueOf(verificationUin).toString();
    }

    @Required
    public void setDnsVerificationsService(DNSVerificationsService dnsVerificationsService) {
        this.dnsVerificationsService = dnsVerificationsService;
    }

    @Required
    public void setDnsCacheTtl(int dnsCacheTtl) {
        this.dnsCacheTtl = dnsCacheTtl;
    }

    @Required
    public void setDnsNegativeCacheTtl(int dnsNegativeCacheTtl) {
        this.dnsNegativeCacheTtl = dnsNegativeCacheTtl;
    }
}
