package ru.yandex.wmconsole.verification;

import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Random;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.wmconsole.data.VerificationStateEnum;
import ru.yandex.wmconsole.data.info.UsersHostsInfo;
import ru.yandex.wmtools.common.SupportedProtocols;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.sita.DocumentFormatEnum;
import ru.yandex.wmtools.common.sita.SitaException;
import ru.yandex.wmtools.common.sita.SitaRequestTimeout;
import ru.yandex.wmtools.common.sita.SitaService;
import ru.yandex.wmtools.common.sita.SitaUrlFetchRequest;
import ru.yandex.wmtools.common.sita.SitaUrlFetchRequestBuilder;
import ru.yandex.wmtools.common.sita.SitaUrlFetchResponse;
import ru.yandex.wmtools.common.sita.UserAgentEnum;
import ru.yandex.wmtools.common.util.http.YandexHttpStatus;

/**
 * @author ailyin
 */
public class TxtFileVerifier extends ConsistentVerifier {
    private static final Logger log = LoggerFactory.getLogger(TxtFileVerifier.class);

    private static final int PAUSE_TO_PREVENT_HOST_OVERLOADING = 1000;

    private int socketTimeoutMillis = 10000;

    private SitaService sitaNoCheckService;

    /**
     * Checks, whether given urlAddress exists and has zero length.
     *
     * @param verInfo ...
     * @return Returns whether urlAddress exists and has zero length.
     */
    @Override
    public UsersHostsInfo verify(UsersHostsInfo verInfo) {
//        log.debug("Txt file check disabled");
//        return UsersHostsInfo.createNotVerifiedNow(verInfo,
//                VerificationStateEnum.TXT_FILE_METHOD_NOT_APPLICABLE_FOR_HOST,
//                "Txt file check disabled");
        // TODO: disabled du to security bug WMC-1125
        // Reenabled for periodic checks
        UsersHostsInfo result = verifyRandomTxtFiles(verInfo);
        if (result != null) {
            return result;
        }

        try {
            URL urlAddress = getTxtFileVerificationAddress(verInfo.getHostName(), verInfo.getVerificationUin());
            log.debug("Verifying " + urlAddress + "...");
            result = checkTxtFile(urlAddress, verInfo);
            log.debug("Verification result: " + verInfo.getVerifyFaultLog());
        } catch (SitaException e) {
            String verifyFaultLog = "Not verified: Unable to download file.\n" + e;
            log.debug(verifyFaultLog, e);
            return UsersHostsInfo.createNotVerifiedNow(verInfo, VerificationStateEnum.TXT_FILE_IO_EXCEPTION, verifyFaultLog);
        } catch (InternalException e) {
            String verifyFaultLog = "Not verified: Unable to download file.\n" + e;
            log.debug(verifyFaultLog, e);
            return UsersHostsInfo.createNotVerifiedNow(verInfo, VerificationStateEnum.TXT_FILE_IO_EXCEPTION, verifyFaultLog);
        }

        UsersHostsInfo tempRes = verifyRandomTxtFiles(verInfo);
        if (tempRes != null) {
            return tempRes;
        }

        return result;
    }

    private UsersHostsInfo verifyRandomTxtFiles(UsersHostsInfo verInfo) {
        Random random = new Random(System.currentTimeMillis());

        // several requests to check that host does not always return HTTP OK code.
        for (int i = 0; i < 1; i++) {
            boolean errorFound = false;
            try {
                URL urlAddress = getTxtFileVerificationAddress(verInfo.getHostName(), random.nextLong());
                log.debug("Checking " + urlAddress + "...");
                UsersHostsInfo info = checkTxtFile(urlAddress, verInfo);
                if (info.getVerificationState() == null || !info.getVerificationState().isVerified()) {
                    log.debug("Successful:" + info.getVerifyFaultLog());
                    errorFound = true;
                }
            } catch (InternalException e) {
                log.error("Unable to verify host", e);
                errorFound = true;
            } catch (SitaException e) {
                log.error("Unable to verify host", e);
                errorFound = true;
            }
            if (!errorFound) {
                String verifyFaultLog = "Not verified: TXT-File method is not applicable for this host.";
                log.debug(verifyFaultLog);
                return UsersHostsInfo.createNotVerifiedNow(verInfo, VerificationStateEnum.TXT_FILE_METHOD_NOT_APPLICABLE_FOR_HOST, verifyFaultLog);
            }
        }

        return null;
    }

    private UsersHostsInfo checkTxtFile(URL urlAddress, UsersHostsInfo verInfo)
            throws InternalException
    {
        try {
            Thread.sleep(PAUSE_TO_PREVENT_HOST_OVERLOADING);
        } catch (InterruptedException e) {
            log.error("Strange interrupt exception: waiting failed", e);
        }

        SitaUrlFetchRequest sitaUrlFetchRequest = new SitaUrlFetchRequestBuilder(urlAddress)
                .setUserAgent(UserAgentEnum.WEBMASTER)
                .setCheckAllowedInRobotsTxt(false)
                .setDocumentFormat(DocumentFormatEnum.DF_NO_DOC)
                .setRequestTimeout(SitaRequestTimeout._15_SECONDS)
                .createSitaUrlFetchRequest();
        sitaUrlFetchRequest.setSitaSocketTimeoutMillis(socketTimeoutMillis);
        SitaUrlFetchResponse sitaUrlFetchResponse = sitaNoCheckService.request(sitaUrlFetchRequest);

        YandexHttpStatus status = sitaUrlFetchResponse.getSitaHttpStatus();
        if (status != YandexHttpStatus.HTTP_200_OK) {
            if (YandexHttpStatus.isStandardHttpCode(status)) {
                String verifyFaultLog = "Not verified: Unable to download file because bad http code " + status.getCode() + "\n";
                return UsersHostsInfo.createNotVerifiedNow(verInfo, VerificationStateEnum.TXT_FILE_IO_EXCEPTION, verifyFaultLog);
            } else {
                throw new InternalException(InternalProblem.PROCESSING_ERROR, "Unable to download file, status = " + status);
            }
        }

        log.debug("Verified successfully");
        return UsersHostsInfo.createVerifiedNow(verInfo);
    }

    /**
     * Given the host name and verification uin, this method creates and returns full URL Address, where
     * verification file must be situated.
     *
     * @param hostname        hostname to verificate.
     * @param verificationUin verification uin for given host and given user.
     * @return Returns full URL Address, where verification file must be situated.
     * @throws IllegalArgumentException if invalid hostname has been passed
     */
    private URL getTxtFileVerificationAddress(String hostname, long verificationUin) {
        try {
            String url = String.format("%1$s/yandex_%2$s.txt",
                    hostname.toLowerCase(), VerificationIdentifier.valueOf(verificationUin).toString());
            return SupportedProtocols.getURL(url);
        } catch (MalformedURLException e) {
            throw new IllegalArgumentException("invalid hostname has been passed", e);
        } catch (URISyntaxException e) {
            throw new IllegalArgumentException("invalid hostname has been passed", e);
        } catch (SupportedProtocols.UnsupportedProtocolException e) {
            throw new IllegalArgumentException("hostname with unsupported protocol has been passed", e);
        }
    }

    @Required
    public void setSitaNoCheckService(SitaService sitaService) {
        this.sitaNoCheckService = sitaService;
    }
}
