package ru.yandex.wmconsole.verification;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import ru.yandex.misc.enums.IntEnum;
import ru.yandex.misc.enums.IntEnumResolver;

public enum VerificationTypeEnum implements IntEnum{
    /**
     * Default verification type is used in database when host is not verified, or when it is verified by support
     */
    TXT_FILE(    0, VerificationMoment.ON_USER_DEMAND,      0),
    META_TAG(    1, VerificationMoment.ON_USER_DEMAND,      0),
    CHEAT(       2, VerificationMoment.ON_EXTERNAL_DEMAND,  0),
    AUTO(        3, VerificationMoment.ON_ADDING_HOST,      2, false, false),
    MANUAL(      4, VerificationMoment.ON_EXTERNAL_DEMAND,  0),
    PDD_EXTERNAL(5, VerificationMoment.ON_EXTERNAL_DEMAND,  0, true, true),
    PDD_AUTO(    6, VerificationMoment.ON_ADDING_HOST,      1, true, true),
    HTML_FILE(   7, VerificationMoment.ON_USER_DEMAND,      0),
    DNS_RECORD(  8, VerificationMoment.ON_USER_DEMAND,      0),
    DELEGATION(  9, VerificationMoment.ON_ADDING_HOST,      3),
    WHOIS(      10, VerificationMoment.ON_USER_DEMAND,      0),
    ;

    /**
     * Default verification type, used when real verification type does not exist
     * or does not matter.
     */
    public static final VerificationTypeEnum DEFAULT = META_TAG;

    private final int value;
    private final VerificationMoment moment;
    private final int priority;     // used for auto verifications only
    private final boolean possibleToCancel;
    private final boolean external;

    private VerificationTypeEnum(int value, VerificationMoment moment,
                                 int priority) {
        this(value, moment, priority, true, false);
    }

    private VerificationTypeEnum(int value, VerificationMoment moment,
                                 int priority, boolean possibleToCancel, boolean external) {
        this.value = value;
        this.moment = moment;
        this.priority = priority;
        this.possibleToCancel = possibleToCancel;
        this.external = external;
    }

    public int value() {
        return value;
    }

    public VerificationMoment getMoment() {
        return moment;
    }

    public boolean isPossibleToCancel() {
        return possibleToCancel;
    }

    public boolean isExternal() {
        return external;
    }

    public static final IntEnumResolver<VerificationTypeEnum> R = IntEnumResolver.r(VerificationTypeEnum.class);

    /**
     * Returns list of verification methods applied when user is adding host. Methods are listed in order of descending priority.
     *
     * @return verification methods applied when user is adding host
     */
    public static List<VerificationTypeEnum> getAutoVerifications() {
        List<VerificationTypeEnum> result = new ArrayList<VerificationTypeEnum>();
        for (VerificationTypeEnum type : values()) {
            if (VerificationMoment.ON_ADDING_HOST.equals(type.getMoment())) {
                result.add(type);
            }
        }
        Collections.sort(result, new Comparator<VerificationTypeEnum>() {
            //Note: this comparator imposes orderings that are inconsistent with equals.
            @Override
            public int compare(VerificationTypeEnum type1, VerificationTypeEnum type2) {
                return type2.priority - type1.priority;
            }
        });
        return result;
    }
}
