package ru.yandex.wmconsole.notifier.handler;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.common.util.collections.Cu;
import ru.yandex.common.util.functional.Fu;
import ru.yandex.webmaster.common.host.dao.TblHostsMainDao;
import ru.yandex.webmaster.common.host.dao.TblUsersHostsDao;
import ru.yandex.wmconsole.data.NotificationTypeEnum;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.service.BadSoftwareNotificationService;
import ru.yandex.wmconsole.service.NotificationService;
import ru.yandex.wmtools.common.SupportedProtocols;
import ru.yandex.wmtools.common.error.InternalException;

import java.io.IOException;
import java.io.StringReader;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * User: azakharov
 * Date: 08.08.14
 * Time: 12:42
 */
public class BadSoftwareNotificationHandler implements Handler {

    private Logger log = LoggerFactory.getLogger(BadSoftwareNotificationHandler.class);

    private TblHostsMainDao tblHostsMainDao;
    private TblUsersHostsDao tblUsersHostsDao;

    private BadSoftwareNotificationService badSoftwareNotificationService;
    private NotificationService notificationService;

    @Override
    public void handleNotification(final String xmlData) {

        log.info("Bad software notification received externally");
        List<BadSoftwareHost> hosts = parseXmlData(xmlData);

        for (BadSoftwareHost host : hosts) {
            try {
                URL hostUrl = SupportedProtocols.getURL(host.getHostName());
                String hostName = SupportedProtocols.getCanonicalHostname(hostUrl);

                BriefHostInfo hostInfo = tblHostsMainDao.getHostIdByHostname(hostName);
                if (hostInfo == null) {
                    continue;
                }

                List<Long> users = tblUsersHostsDao.getVerifiedUserByHosts(hostInfo.getId());
                if (users.isEmpty()) {
                    continue;
                }

                Date receiveTime = new Date();
                for (String url : host.getUrls()) {
                    Long issueId = badSoftwareNotificationService.saveNotificationInfo(
                            hostInfo.getId(), url, receiveTime);
                    notificationService.insertNotificationForUsers(NotificationTypeEnum.NEW_BAD_SOFTWARE, issueId, users, receiveTime);
                }
            } catch (MalformedURLException e) {
                log.error("Malformed URL " + host.getHostName(), e);
            } catch (URISyntaxException e) {
                log.error("URI syntax error " + host.getHostName(), e);
            } catch (SupportedProtocols.UnsupportedProtocolException e) {
                log.error("Unsupported protocol " + host.getHostName(), e);
            } catch (InternalException e) {
                log.error("Internal exception " + host.getHostName(), e);
            }
        }
    }

    private List<BadSoftwareHost> parseXmlData(final String xmlData) {
        StringReader stringReader = new StringReader(xmlData);
        SAXBuilder builder = new SAXBuilder();
        try {
            Document document = builder.build(stringReader);
            Element rootElement = document.getRootElement();
            List<BadSoftwareHost> result = Cu.map(createHost(textOf()), rootElement.getChildren("host"));
            log.debug("Bad software host list size is " + result.size());
            return result;
        } catch (JDOMException e) {
            log.error("JDOMException in " + getClass().getName() + " " + "while extracting xml data", e);
            return Collections.emptyList();
        } catch (IOException e) {
            log.error("IOException in " + getClass().getName() + " " + "while reading xml data", e);
            return Collections.emptyList();
        } catch (NullPointerException e) {
            log.warn("Invalid xml notification data format", e);
            return Collections.emptyList();
        }
    }

    private static Fu<Element, String> textOf() {
        return new Fu<Element, String>() {
            @Override
            public String apply(Element element) {
                return element.getText();
            }
        };
    }

    private static Fu<Element, BadSoftwareHost> createHost(final Fu<Element, String> textOf) {
        return new Fu<Element, BadSoftwareHost>() {
            @Override
            public BadSoftwareHost apply(Element hostElement) {
                String hostName = hostElement.getAttributeValue("name");
                List<String> urls = Cu.map(textOf, hostElement.getChildren("url"));
                return new BadSoftwareHost(hostName, urls);
            }
        };
    }

    @Override
    public void internalHandle(String... params) {
        throw new UnsupportedOperationException("Internal handle not supported for bad software notification");
    }

    @Override
    public void handleInternalNotification(Map<String, String> params) {
        throw new UnsupportedOperationException("Internal handle not supported for bad software notification");
    }

    public static class BadSoftwareHost {

        private final String hostName;
        private final List<String> urls;

        public BadSoftwareHost(String hostName, List<String> urls) {
            this.hostName = hostName;
            this.urls = urls;
        }

        public String getHostName() {
            return hostName;
        }

        public List<String> getUrls() {
            return urls;
        }
    }

    @Required
    public void setTblHostsMainDao(TblHostsMainDao tblHostsMainDao) {
        this.tblHostsMainDao = tblHostsMainDao;
    }

    @Required
    public void setTblUsersHostsDao(TblUsersHostsDao tblUsersHostsDao) {
        this.tblUsersHostsDao = tblUsersHostsDao;
    }

    @Required
    public void setBadSoftwareNotificationService(BadSoftwareNotificationService badSoftwareNotificationService) {
        this.badSoftwareNotificationService = badSoftwareNotificationService;
    }

    @Required
    public void setNotificationService(NotificationService notificationService) {
        this.notificationService = notificationService;
    }
}
