package ru.yandex.wmconsole.notifier.handler;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.common.util.collections.Pair;
import ru.yandex.wmconsole.data.NotificationTypeEnum;
import ru.yandex.wmconsole.service.NotificationService;
import ru.yandex.wmconsole.service.dao.TblMainMirrorNotificationPairDao;
import ru.yandex.wmconsole.service.dao.TblNotificationMirrorGroupsDao;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.servantlet.AbstractServantlet;
import ru.yandex.wmtools.common.util.URLUtil;

import java.io.IOException;
import java.io.StringReader;
import java.net.URL;
import java.util.*;

/**
 * User: azakharov
 * Date: 11.03.14
 * Time: 15:14
 */
public class MirrorGroupsNotificationHandler implements Handler {

    private static Logger log = LoggerFactory.getLogger(MirrorGroupsNotificationHandler.class);

    private static String TAG_GROUPS = "groups";
    private static String TAG_HOST = "host";
    private static String ATTR_NAME = "name";
    private static String ATTR_NEW_MAIN_MIRROR = "new-main-mirror";

    private TblMainMirrorNotificationPairDao tblMainMirrorNotificationPairDao;
    private TblNotificationMirrorGroupsDao tblNotificationMirrorGroupsDao;
    private NotificationService notificationService;

    @Override
    public void handleNotification(String xmlData) {

        log.debug("Change main mirror groups notification received externally");

        Date now = new Date();

        try {
            List<Pair<String, String>> pairs = handleXML(xmlData);

            Map<String, String> hostMainMirror = new HashMap<>();
            for (Pair<String, String> p : pairs) {
                try {
                    URL host = AbstractServantlet.prepareUrl(p.getFirst(), true);
                    URL newMainMirror = AbstractServantlet.prepareUrl(p.getSecond(), true);

                    String hostName = URLUtil.getHostName(host, false);
                    String newMirrorName = URLUtil.getHostName(newMainMirror, false);

                    hostMainMirror.put(hostName, newMirrorName);
                } catch (UserException e) {
                    throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "Attempting to change main mirror from " + p.getFirst() + " to " + p.getSecond() + ". " +
                            "new hostname is invalid");
                }
            }

            Long mirrorGroupsId = tblNotificationMirrorGroupsDao.insertNotification();
            if (mirrorGroupsId == null) {
                log.error("insert to tbl_notification_mirrorgroups failed");
                return;
            }

            for (Map.Entry<String, String> entry : hostMainMirror.entrySet()) {
                log.debug("Change main mirror for host " + entry.getKey() + " to " + entry.getValue());
                tblMainMirrorNotificationPairDao.savePair(mirrorGroupsId, entry.getKey(), entry.getValue());
            }

            notificationService.insertNotificationForUser(NotificationTypeEnum.MIRROR_GROUPS, mirrorGroupsId, 0L, now);
        } catch (InternalException e) {
            log.error("Exception during notification processing", e);
        }
    }

    private List<Pair<String, String>> handleXML(String xmlData) throws InternalException {
        SAXBuilder builder = new SAXBuilder();
        StringReader stringReader = new StringReader(xmlData);
        try {
            Document eventDocument = builder.build(stringReader);
            Element rootElement = eventDocument.getRootElement();
            if (!TAG_GROUPS.equals(rootElement.getName())) {
                throw new IllegalArgumentException("Bad root element name");
            }
            List<Element> hosts = rootElement.getChildren(TAG_HOST);
            List<Pair<String, String>> result = new ArrayList<>(hosts.size());
            for (Element host : hosts) {
                result.add(new Pair<String, String>(host.getAttributeValue(ATTR_NAME), host.getAttributeValue(ATTR_NEW_MAIN_MIRROR)));
            }
            return result;
        } catch (JDOMException e) {
            log.error("JDOMException in " + getClass().getName() + " " + "while extracting xml data", e);
            throw new InternalException(InternalProblem.ILLEGAL_ARGUMENT, "Invalid xml notification data format");
        } catch (IOException e) {
            log.error("IOException in " + getClass().getName() + " " + "while reading xml data", e);
            throw new InternalException(InternalProblem.ILLEGAL_ARGUMENT, "Invalid xml notification data format");
        }
    }

    @Override
    public void internalHandle(String... params) {
        throw new UnsupportedOperationException("Internal handle is not supported for " +
                NotificationTypeEnum.MIRROR_GROUPS + " notification");
    }

    @Override
    public void handleInternalNotification(Map<String, String> params) {
        throw new UnsupportedOperationException("Internal handle is not supported for " +
                NotificationTypeEnum.MIRROR_GROUPS + " notification");
    }

    @Required
    public void setTblMainMirrorNotificationPairDao(TblMainMirrorNotificationPairDao tblMainMirrorNotificationPairDao) {
        this.tblMainMirrorNotificationPairDao = tblMainMirrorNotificationPairDao;
    }

    @Required
    public void setTblNotificationMirrorGroupsDao(TblNotificationMirrorGroupsDao tblNotificationMirrorGroupsDao) {
        this.tblNotificationMirrorGroupsDao = tblNotificationMirrorGroupsDao;
    }

    @Required
    public void setNotificationService(NotificationService notificationService) {
        this.notificationService = notificationService;
    }
}
