package ru.yandex.wmconsole.servantlet;

import org.springframework.beans.factory.annotation.Required;
import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.common.util.xml.XmlConvertable;
import ru.yandex.wmconsole.data.info.ShortHostInfo;
import ru.yandex.wmconsole.service.ConsistentMainMirrorService;
import ru.yandex.wmconsole.service.UsersHostsService;
import ru.yandex.wmconsole.util.WwwUtil;
import ru.yandex.wmconsole.util.XmlUtil;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.servantlet.AbstractServantlet;
import ru.yandex.wmtools.common.util.XmlConvertableCollectionWrapper;

import javax.xml.bind.JAXBException;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import java.net.IDN;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * Check verification status for given user and list of hosts. CHEAT verifications are excluded.
 * host names must be in ascii (punycode for IDN) and in lower case
 * @author senin
 */
public class CheckHostsVerifiedServantlet extends AbstractServantlet {

    private static final String PARAM_UID = "uid";
    private static final String PARAM_HOST = "host";
    private static final String PARAM_ALLOW_WWW = "allow-www";

    private UsersHostsService usersHostsService;
    private ConsistentMainMirrorService consistentMainMirrorService;

    @Override
    protected void doProcess(ServRequest req, final ServResponse res) throws UserException, InternalException {

        Long userId = getRequiredLongParam(req, PARAM_UID);
        List<String> hosts = req.getMultiParams(PARAM_HOST);
        Boolean allowWww = getBooleanParam(req, PARAM_ALLOW_WWW);
        final List<? extends ShortHostInfo> hostsOfUsers;
        final Map<String, String> mainToOriginal = new HashMap<String, String>();
        final Map<String, String> switchedWwwToOriginal = new HashMap<String, String>();
        final List<ShortHostInfo> wwwSwitchedHostsOfUsers;
        if (hosts.isEmpty()) {
            // Если хосты не указаны, показываем все хосты пользователя
            hostsOfUsers = usersHostsService.getUserHosts(userId, false);
            wwwSwitchedHostsOfUsers = Collections.emptyList();
        } else {
            // Если хосты указаны, показываем информацию по хостам пользователя среди указанных

            List<String> mainMirrors = new ArrayList<>(hosts.size());

            for (String hostName : hosts) {
                String mainMirror = consistentMainMirrorService.getMainMirror(hostName);
                mainMirrors.add(mainMirror);
                mainToOriginal.put(mainMirror, hostName);
            }


            List<String> wwwSwitched = new LinkedList<String>();
            for (String h : hosts) {
                String wwwTwin = WwwUtil.switchWWW(h);
                wwwSwitched.add(wwwTwin);
                switchedWwwToOriginal.put(wwwTwin, h);
            }

            hostsOfUsers = usersHostsService.checkHostsVerifiedByUser(
                    userId, mainMirrors.toArray(new String[mainMirrors.size()]));

            wwwSwitchedHostsOfUsers = usersHostsService.checkHostsVerifiedByUser(
                    userId, wwwSwitched.toArray(new String[wwwSwitched.size()]));
        }

        List<XmlConvertable> wrappers = new LinkedList<XmlConvertable>();
        addWrappers(hostsOfUsers, mainToOriginal, wrappers);

        if (allowWww != null && allowWww) {
            addWrappers(wwwSwitchedHostsOfUsers, switchedWwwToOriginal, wrappers);
        }

        res.addData(new XmlConvertableCollectionWrapper(wrappers, "hosts"));
    }

    public static enum HostStateEnum {
        ADDED_NOT_VERIFIED,
        VERIFIED
    }

    private void addWrappers(List<? extends ShortHostInfo> hostsOfUsers, Map<String, String> originalNameMap, List<XmlConvertable> wrappers) throws InternalException {
        for (ShortHostInfo info : hostsOfUsers) {
            final HostStateEnum simpleState;
            switch (info.getVerificationState()) {
                case VERIFIED:
                    simpleState = HostStateEnum.VERIFIED;
                    break;
                default:
                    simpleState = HostStateEnum.ADDED_NOT_VERIFIED;
            }
            String original = originalNameMap.get(info.getName());
            if (original == null) {
                original = info.getName();
            }
            try {
                wrappers.add(XmlUtil.xmlConvertable(
                        new HostStateInfo(
                                info.getId(),
                                info.getName(),
                                original,
                                IDN.toASCII(original),
                                simpleState)));
            } catch (JAXBException e) {
                throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "JAXB exception", e);
            }
        }
    }

    @XmlRootElement(name = "host")
    public static class HostStateInfo {
        private long hostId;
        private String name;
        private String originalName;
        private String originalPunicodeName;
        private HostStateEnum hostState;

        public HostStateInfo() {
            // for JAXB
        }

        public HostStateInfo(long hostId, String name, String originalName, String originalPunicodeName, HostStateEnum hostState) {
            this.hostId = hostId;
            this.name = name;
            this.originalName = originalName;
            this.originalPunicodeName = originalPunicodeName;
            this.hostState = hostState;
        }

        @XmlAttribute(name = "id")
        public long getHostId() {
            return hostId;
        }

        @XmlElement(name = "name")
        public String getName() {
            return name;
        }

        @XmlElement(name = "original-name")
        public String getOriginalName() {
            return originalName;
        }

        @XmlElement(name = "original-punicode-name")
        public String getOriginalPunicodeName() {
            return originalPunicodeName;
        }

        @XmlElement(name = "host-state")
        public HostStateEnum getHostState() {
            return hostState;
        }
    }

    @Required
    public void setUsersHostsService(UsersHostsService usersHostsService) {
        this.usersHostsService = usersHostsService;
    }

    @Required
    public void setConsistentMainMirrorService(ConsistentMainMirrorService consistentMainMirrorService) {
        this.consistentMainMirrorService = consistentMainMirrorService;
    }
}
