package ru.yandex.wmconsole.servantlet;

import java.util.*;

import javax.xml.bind.JAXBException;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.common.util.xml.XmlConvertable;
import ru.yandex.wmconsole.data.info.UsersHostsInfo;
import ru.yandex.wmconsole.service.ConsistentMainMirrorService;
import ru.yandex.wmconsole.service.HostInfoService;
import ru.yandex.wmconsole.service.UsersHostsService;
import ru.yandex.wmconsole.util.XmlUtil;
import ru.yandex.wmconsole.verification.VerificationTypeEnum;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.error.UserProblem;
import ru.yandex.wmtools.common.servantlet.AbstractServantlet;
import ru.yandex.wmtools.common.util.XmlConvertableCollectionWrapper;

/**
 * Сервис для получения списка пользователей, для которых подтвержден заданный сайт.
 *
 * User: Alexey Zakharov <azakharov@yandex-team.ru>
 * Date: 08.02.12
 */
public class HostUsersServantlet extends AbstractServantlet {
    private static final Logger log = LoggerFactory.getLogger(HostUsersServantlet.class.getName());

    private static final String PARAM_HOST = "host";

    private UsersHostsService usersHostsService;
    private HostInfoService hostInfoService;
    private ConsistentMainMirrorService consistentMainMirrorService;

    @Override
    protected void doProcess(ServRequest req, final ServResponse res) throws UserException, InternalException {

        final List<String> hostNames = req.getMultiParams(PARAM_HOST);
        if (hostNames.isEmpty()) {
            log.error("Empty parameter " + PARAM_HOST);
            throw new UserException(UserProblem.REQUIRED_PARAM_MISSED, "Required param missed: " + PARAM_HOST, PARAM_HOST);
        }

        try {
            final List<XmlConvertable> wrappers = new LinkedList<XmlConvertable>();
            final List<String> mainMirrors = new ArrayList<>(hostNames.size());
            for (String hostName : hostNames) {
                mainMirrors.add(consistentMainMirrorService.getMainMirror(hostName));
            }
            final Iterator<String> iterator = mainMirrors.iterator();
            for (String hostName : hostNames) {
                final String mainMirror = iterator.next();
                final Long hostId = hostInfoService.getHostIdByHostName(mainMirror, true);
                if (hostId == null) {
                    // Хоста нет в tbl_hosts
                    continue;
                }
                final List<UsersHostsInfo> usersHostsInfos = usersHostsService.listVerifiedUsersForHost(hostId);
                final List<UserInfo> userInfos = new LinkedList<UserInfo>();
                for (UsersHostsInfo info : usersHostsInfos) {
                    final VerificationTypeEnum vt = info.getVerificationType();
                    if (vt == null) {
                        continue;
                    }
                    userInfos.add(new UserInfo(
                            info.getUserId(),
                            vt,
                            info.getVerificationDate()));
                }
                final HostUsersInfo hostUsersInfo = new HostUsersInfo(hostName, userInfos);
                final XmlConvertable wrapper = XmlUtil.xmlConvertable(hostUsersInfo);
                wrappers.add(wrapper);
            }
            res.addData(new XmlConvertableCollectionWrapper(wrappers, "hosts"));
        } catch (JAXBException e) {
            log.error("Exception during xml marshalling", e);
            throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "Exception during xml marshalling");
        }
    }

    @XmlRootElement(name = "host")
    public static class HostUsersInfo {
        private String hostName;
        private List<UserInfo> users;

        public HostUsersInfo() {
            // for JAXB
        }

        public HostUsersInfo(String hostName, List<UserInfo> users) {
            this.hostName = hostName;
            this.users = users;
        }

        @XmlAttribute(name = "name", required = true)
        public String getHostName() {
            return hostName;
        }

        @XmlElement(name = "user-info", required = true)
        public List<UserInfo> getUsers() {
            return users;
        }
    }

    public static class UserInfo {
        private long uid;
        private VerificationTypeEnum verificationType;
        private Date verificationDate;

        public UserInfo() {
            // for JAXB
        }

        public UserInfo(long uid, VerificationTypeEnum verificationType, Date verificationDate) {
            this.uid = uid;
            this.verificationType = verificationType;
            this.verificationDate = verificationDate;
        }

        @XmlAttribute(name = "user-id", required = true)
        public long getUid() {
            return uid;
        }

        @XmlAttribute(name = "type", required = true)
        public int getVerificationType() {
            return verificationType.value();
        }

        @XmlAttribute(name = "date", required = true)
        public Date getVerificationDate() {
            return verificationDate;
        }
    }

    @Required
    public void setUsersHostsService(UsersHostsService usersHostsService) {
        this.usersHostsService = usersHostsService;
    }

    @Required
    public void setHostInfoService(HostInfoService hostInfoService) {
        this.hostInfoService = hostInfoService;
    }

    @Required
    public void setConsistentMainMirrorService(ConsistentMainMirrorService consistentMainMirrorService) {
        this.consistentMainMirrorService = consistentMainMirrorService;
    }
}
