package ru.yandex.wmconsole.servantlet.internal.api;

import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.util.collections.Pair;
import ru.yandex.webmaster.common.http.Action;
import ru.yandex.webmaster.common.http.ActionResponse;
import ru.yandex.webmaster.common.http.RequestPostProperty;
import ru.yandex.wmconsole.service.ConsistentMainMirrorService;
import ru.yandex.wmconsole.service.VerifyViewerService;
import ru.yandex.wmtools.common.servantlet.AbstractServantlet;
import ru.yandex.wmtools.common.util.URLUtil;

/**
 * User: azakharov
 * Date: 18.04.14
 * Time: 19:34
 */
public class CheckVerificationAction extends Action<CheckVerificationAction.CheckVerificationRequest, CheckVerificationAction.CheckVerificationResponse> {

    private static Logger log = LoggerFactory.getLogger(CheckVerificationAction.class);
    private VerifyViewerService verifyService;
    private ConsistentMainMirrorService consistentMainMirrorService;

    @Override
    public CheckVerificationResponse process(final CheckVerificationRequest request) throws Exception {

        // prepare input data
        final List<Long> uids = new ArrayList<Long>();
        final List<String> hostNames = new ArrayList<String>();

        int ignoredPairs = 0;
        for (UserHostPair p : request.getUsersHosts()) {
            if (p.userId != null && !StringUtils.isEmpty(p.getHostName())) {
                try {
                    URL url = AbstractServantlet.doPrepareUrl(p.getHostName(), true, true);
                    hostNames.add(URLUtil.getHostName(url, false));
                    uids.add(p.getUserId());
                } catch (Exception e) {
                    log.error("user host pair is incorrect: {} {}", p.userId, p.getHostName());
                    ignoredPairs++;
                }
            } else {
                log.warn("user host pair is not complete: {} {}", p.userId, p.getHostName());
                ignoredPairs++;
            }
        }

        // convert hostName to main mirror if necessary
        final List<String> hostNamesForCheck;
        if (request.isConvertToMainMirror()) {
            hostNamesForCheck = new ArrayList<>(hostNames.size());
            for (String hostName : hostNames) {
                hostNamesForCheck.add(consistentMainMirrorService.getMainMirror(hostName));
            }
        } else {
            hostNamesForCheck = hostNames;
        }

        final List<Pair<Long, String>> verifications = new ArrayList<Pair<Long, String>>();
        for (int i = 0; i < hostNamesForCheck.size(); i++) {
            verifications.add(new Pair<Long, String>(uids.get(i), hostNamesForCheck.get(i)));
        }

        final Collection<Pair<Long, String>> verifiedList = verifyService.checkVerificationForUsersAndHosts(verifications);
        final Set<UserHostPair> verifiedCaseInsensitiveSet = new HashSet<UserHostPair>();
        for (Pair<Long, String> pair : verifiedList) {
            verifiedCaseInsensitiveSet.add(new UserHostPair(pair));
        }
        log.debug("To verify list size: " + verifications.size());
        log.debug("Verified: " + verifiedList.size());

        final List<UserHostVerificationInfo> verificationInfos = new ArrayList<>(verifications.size());
        for (int i = 0; i < verifications.size(); i++) {
            final Pair<Long, String> userHostPair = verifications.get(i);
            final boolean isVerified = verifiedCaseInsensitiveSet.contains(new UserHostPair(userHostPair));
            verificationInfos.add(new UserHostVerificationInfo(userHostPair, isVerified));
        }

        return new CheckVerificationResponse(verificationInfos, ignoredPairs);
    }

    public static class CheckVerificationRequest implements InternalApiRequest {
        private String source;
        private UserHostPair[] usersHosts;
        private boolean convertToMainMirror = false;

        @Override
        public String getSource() {
            return source;
        }

        @Override
        public void setSource(String source) {
            this.source = source;
        }


        @RequestPostProperty(required = true)
        public void setUsersHosts(UserHostPair[] usersHosts) {
            this.usersHosts = usersHosts;
        }

        public UserHostPair[] getUsersHosts() {
            return usersHosts;
        }

        public boolean isConvertToMainMirror() {
            return convertToMainMirror;
        }

        @RequestPostProperty
        public void setConvertToMainMirror(boolean convertToMainMirror) {
            this.convertToMainMirror = convertToMainMirror;
        }
    }

    public static class UserHostPair {
        private Long userId;
        private String hostName;

        /**
         * Constructor for request parsing and reflection
         */
        public UserHostPair() {
        }

        /**
         * Constructor for usage from user code
         */
        public UserHostPair(Pair<Long,String> pair) {
            this.userId = pair.getFirst();
            this.hostName = pair.getSecond();
        }

        public Long getUserId() {
            return userId;
        }

        public void setUserId(Long userId) {
            this.userId = userId;
        }

        public String getHostName() {
            return hostName;
        }

        public void setHostName(String hostName) {
            this.hostName = hostName;
        }

        @Override
        public int hashCode() {
            int result = userId != null ? userId.hashCode() : 0;
            result = 31 * result + (hostName != null ? hostName.toLowerCase().hashCode() : 0);
            return result;
        }

        @Override
        public boolean equals(Object o) {
            return o instanceof UserHostPair && hostName.equalsIgnoreCase(((UserHostPair) o).getHostName()) && userId.equals(((UserHostPair) o).getUserId());
        }
    }

    public static class CheckVerificationResponse implements ActionResponse {
        private final List<UserHostVerificationInfo> usersHosts;
        private final int ignoredUserHostPairsCount;

        public CheckVerificationResponse(final List<UserHostVerificationInfo> usersHosts, final int ignoredUserHostPairsCount) {
            this.usersHosts = usersHosts;
            this.ignoredUserHostPairsCount = ignoredUserHostPairsCount;
        }

        public List<UserHostVerificationInfo> getUsersHosts() {
            return usersHosts;
        }

        public int getIgnoredUserHostPairsCount() {
            return ignoredUserHostPairsCount;
        }
    }

    public static class UserHostVerificationInfo extends UserHostPair {
        private final boolean verified;

        public UserHostVerificationInfo(final Pair<Long, String> userHostPair, final boolean verified) {
            super(userHostPair);
            this.verified = verified;
        }

        public boolean isVerified() {
            return verified;
        }
    }

    @Required
    public void setVerifyService(VerifyViewerService verifyService) {
        this.verifyService = verifyService;
    }

    @Required
    public void setConsistentMainMirrorService(ConsistentMainMirrorService consistentMainMirrorService) {
        this.consistentMainMirrorService = consistentMainMirrorService;
    }
}
