package ru.yandex.wmconsole.servantlet.internal.api.verification;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.webmaster.common.host.dao.TblUsersHostsDao;
import ru.yandex.webmaster.common.http.Action;
import ru.yandex.webmaster.common.http.ActionRequest;
import ru.yandex.webmaster.common.http.ActionResponse;
import ru.yandex.webmaster.common.http.RequestQueryProperty;
import ru.yandex.wmconsole.data.VerificationStateEnum;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.service.AddHostService;
import ru.yandex.wmconsole.service.HostInfoService;
import ru.yandex.wmconsole.util.BackportedIdUtils;
import ru.yandex.wmconsole.util.BackportedWebmasterHostId;
import ru.yandex.wmconsole.verification.VerificationTypeEnum;
import ru.yandex.wmtools.common.service.IViewerUserIdService;

import java.net.URL;
import java.sql.Date;
import java.util.Collections;

/**
 * @author avhaliullin
 */
public class ImportHostVerificationStateAction extends Action<ImportHostVerificationStateAction.Request, ImportHostVerificationStateAction.Response> {
    private static final Logger log = LoggerFactory.getLogger(ImportHostVerificationStateAction.class);

    private HostInfoService hostInfoService;
    private TblUsersHostsDao tblUsersHostsDao;
    private AddHostService addHostService;
    private IViewerUserIdService userIdService;

    @Override
    public Response process(Request request) throws Exception {
        BackportedWebmasterHostId hostId = BackportedIdUtils.stringToHostId(request.hostId);
        String hostName = BackportedIdUtils.toRobotHostString(hostId);
        if (request.uin != null && request.state != null) {
            userIdService.assureUserIsInUsersList(request.userId);
            VerificationTypeEnum type = request.type;
            if (type == null) {
                if (request.state != VerificationStateEnum.VERIFIED) {
                    type = VerificationTypeEnum.META_TAG;
                } else {
                    throw new RuntimeException("Fatal error! Host is verified, but verification type is unknown. Host " + hostId + " user " + request.userId);
                }
            }
            BriefHostInfo hostInfo = hostInfoService.getBriefHostInfoByIdOrName(hostName);
            if (hostInfo != null && tblUsersHostsDao.isAdded(request.userId, hostInfo)) {
                log.info("Will update verification state for user {} host {} directly in users_hosts", request.userId, request.hostId);
                tblUsersHostsDao.addUserHostFromNewWmc(
                        request.userId,
                        hostInfo.getId(),
                        request.state,
                        type,
                        request.uin,
                        new Date(System.currentTimeMillis())
                );
            } else {
                log.info("Launching Big Add Host for user {} host {}", request.userId, request.hostId);
                addHostService.addHostFromNewWebmaster(request.userId, new URL(BackportedIdUtils.hostIdToUrl(hostId)),
                        type, request.state, request.uin);
            }
        } else {
            BriefHostInfo hostInfo = hostInfoService.getBriefHostInfoByIdOrName(hostName);
            if (hostInfo != null) {
                tblUsersHostsDao.deleteHostsForUser(request.userId, Collections.singletonList(hostInfo.getId()));
            }
        }
        return new Response();
    }

    public static class Request implements ActionRequest {
        private long userId;
        private String hostId;
        private VerificationStateEnum state;
        private VerificationTypeEnum type;
        private Long uin;

        @RequestQueryProperty(required = true)
        public void setUserId(long userId) {
            this.userId = userId;
        }

        @RequestQueryProperty(required = true)
        public void setHostId(String hostId) {
            this.hostId = hostId;
        }

        @RequestQueryProperty(required = false)
        public void setState(VerificationStateEnum state) {
            this.state = state;
        }

        @RequestQueryProperty(required = false)
        public void setType(VerificationTypeEnum type) {
            this.type = type;
        }

        @RequestQueryProperty(required = false)
        public void setUin(Long uin) {
            this.uin = uin;
        }
    }

    public static class Response implements ActionResponse {
    }

    @Required
    public void setAddHostService(AddHostService addHostService) {
        this.addHostService = addHostService;
    }

    @Required
    public void setUserIdService(IViewerUserIdService userIdService) {
        this.userIdService = userIdService;
    }

    @Required
    public void setTblUsersHostsDao(TblUsersHostsDao tblUsersHostsDao) {
        this.tblUsersHostsDao = tblUsersHostsDao;
    }

    @Required
    public void setHostInfoService(HostInfoService hostInfoService) {
        this.hostInfoService = hostInfoService;
    }
}
