package ru.yandex.wmconsole.service;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.Date;

import org.springframework.jdbc.core.PreparedStatementCreator;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.jdbc.support.GeneratedKeyHolder;

import ru.yandex.wmconsole.data.SpamNotificationInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.service.AbstractDbService;

/**
 * @author Andrey Mima (amima@yandex-team.ru)
 */
public class SpamNotificationService extends AbstractDbService implements GroupableByHostService<SpamNotificationInfo> {
    private static final String INSERT_SPAM_NOTIFICATION_QUERY =
            "INSERT INTO " +
                        "tbl_notification_spam (url, host_id, receive_time) " +
                    "VALUES (?, ?, ?)";

    private static final String SELECT_SPAM_QUERY =
            "SELECT " +
                        "url, host_id, receive_time " +
                    "FROM " +
                        "tbl_notification_spam " +
                    "WHERE " +
                        "spamnotification_id = ?";

    private static final ParameterizedRowMapper<SpamNotificationInfo> SPAM_NOTIFICATION_MAPPER =
            new ParameterizedRowMapper<SpamNotificationInfo>() {
                private static final String FIELD_URL = "url";
                private static final String FIELD_HOST_ID = "host_id";
                private static final String FIELD_RECEIVE_TIME = "receive_time";

                @Override
                public SpamNotificationInfo mapRow(ResultSet resultSet, int rowNum) throws SQLException {
                    String url = resultSet.getString(FIELD_URL);
                    Long hostId = resultSet.getLong(FIELD_HOST_ID);
                    Date receiveTime = resultSet.getDate(FIELD_RECEIVE_TIME);
                    return new SpamNotificationInfo(url, hostId, receiveTime);
                }
            };

    public Long addSpamNotification(final String url, final Long hostId, final java.util.Date date) throws InternalException {
        GeneratedKeyHolder generatedKeyHolder = new GeneratedKeyHolder();
        PreparedStatementCreator psc = new PreparedStatementCreator() {
            @Override
            public PreparedStatement createPreparedStatement(Connection connection) throws SQLException {
                PreparedStatement ps = connection.prepareStatement(INSERT_SPAM_NOTIFICATION_QUERY,
                        Statement.RETURN_GENERATED_KEYS);
                ps.setString(1, url);
                ps.setLong(2, hostId);
                ps.setTimestamp(3, new Timestamp(date.getTime()));
                return ps;
            }
        };

        getJdbcTemplate(WMCPartition.nullPartition()).getJdbcOperations().update(psc, generatedKeyHolder);
        return generatedKeyHolder.getKey().longValue();
    }

    public SpamNotificationInfo getSpamNotification(Long spamNotificationId) throws InternalException {
        return getJdbcTemplate(WMCPartition.nullPartition()).queryForObject(
                SELECT_SPAM_QUERY,
                SPAM_NOTIFICATION_MAPPER,
                spamNotificationId
        );
    }

    @Override
    public SpamNotificationInfo getSingleNotification(Long id) throws InternalException {
        return getSpamNotification(id);
    }
}
