package ru.yandex.wmconsole.service;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.Date;

import org.springframework.jdbc.core.PreparedStatementCreator;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.jdbc.support.GeneratedKeyHolder;

import ru.yandex.wmconsole.data.VirusEmailNotificationInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.service.AbstractDbService;

/**
 * @author Andrey Mima (amima@yandex-team.ru)
 */
public class VirusNotificationService extends AbstractDbService implements GroupableByHostService<VirusEmailNotificationInfo> {
    private static final String INSERT_VIRUS_NOTIFICATION_QUERY =
            "INSERT INTO " +
                        "tbl_notification_virus (host_id, host_name, email, receive_time) " +
                    "VALUES (?, ?, ?, ?)";

    private static final String SELECT_VIRUS_QUERY =
            "SELECT " +
                        "host_id, host_name, email, receive_time " +
                    "FROM " +
                        "tbl_notification_virus " +
                    "WHERE " +
                        "virus_id = ?";

    private static final ParameterizedRowMapper<VirusEmailNotificationInfo> VIRUS_EMAIL_NOTIFICATION_MAPPER =
            new ParameterizedRowMapper<VirusEmailNotificationInfo>() {
                private static final String FIELD_HOST_ID = "host_id";
                private static final String FIELD_HOST_NAME = "host_name";
                private static final String FIELD_EMAIL = "email";
                private static final String FIELD_RECEIVE_TIME = "receive_time";

                @Override
                public VirusEmailNotificationInfo mapRow(ResultSet resultSet, int rowNum) throws SQLException {
                    Long hostId = resultSet.getLong(FIELD_HOST_ID);
                    String name = resultSet.getString(FIELD_HOST_NAME);
                    String email = resultSet.getString(FIELD_EMAIL);
                    Date receiveTime = resultSet.getDate(FIELD_RECEIVE_TIME);
                    return new VirusEmailNotificationInfo(hostId, name, email, receiveTime);
                }
            };

    public Long addVirusNotification(final Long hostId, final String name, final String email, final java.util.Date date)
            throws InternalException {
        GeneratedKeyHolder generatedKeyHolder = new GeneratedKeyHolder();
        PreparedStatementCreator psc = new PreparedStatementCreator() {
            @Override
            public PreparedStatement createPreparedStatement(Connection connection) throws SQLException {
                PreparedStatement ps = connection.prepareStatement(INSERT_VIRUS_NOTIFICATION_QUERY,
                        Statement.RETURN_GENERATED_KEYS);
                if (hostId != null) {
                    ps.setLong(1, hostId);
                } else {
                    ps.setNull(1, java.sql.Types.BIGINT);
                }
                if (name != null) {
                    ps.setString(2, name);
                } else {
                    ps.setNull(2, java.sql.Types.VARCHAR);
                }
                if (email != null) {
                    ps.setString(3, email);
                } else {
                    ps.setNull(3, java.sql.Types.VARCHAR);
                }
                ps.setTimestamp(4, new Timestamp(date.getTime()));
                return ps;
            }
        };

        getJdbcTemplate(WMCPartition.nullPartition()).getJdbcOperations().update(psc, generatedKeyHolder);
        return generatedKeyHolder.getKey().longValue();
    }

    public VirusEmailNotificationInfo getVirusNotification(Long virusNotificationId) throws InternalException {
        return getJdbcTemplate(WMCPartition.nullPartition()).queryForObject(
                SELECT_VIRUS_QUERY,
                VIRUS_EMAIL_NOTIFICATION_MAPPER,
                virusNotificationId
        );
    }

    @Override
    public VirusEmailNotificationInfo getSingleNotification(Long id) throws InternalException {
        return getVirusNotification(id);
    }
}
