package ru.yandex.wmconsole.service;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.Date;

import org.springframework.jdbc.core.PreparedStatementCreator;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.jdbc.support.GeneratedKeyHolder;

import ru.yandex.wmconsole.data.XssNotificationInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.service.AbstractDbService;

/**
 * @author Andrey Mima (amima@yandex-team.ru)
 */
public class XssNotificationService extends AbstractDbService implements GroupableByHostService<XssNotificationInfo> {
    private static final String INSERT_XSS_NOTIFICATION_QUERY =
            "INSERT INTO " +
                        "tbl_notification_xss (url, host_id, receive_time) " +
                    "VALUES (?, ?, ?)";

    private static final String SELECT_XSS_QUERY =
            "SELECT " +
                        "url, host_id, receive_time " +
                    "FROM " +
                        "tbl_notification_xss " +
                    "WHERE " +
                        "xssnotification_id = ?";

    private static final ParameterizedRowMapper<XssNotificationInfo> XSS_NOTIFICATION_MAPPER =
            new ParameterizedRowMapper<XssNotificationInfo>() {
                private static final String FIELD_URL = "url";
                private static final String FIELD_HOST_ID = "host_id";
                private static final String FIELD_RECEIVE_TIME = "receive_time";

                @Override
                public XssNotificationInfo mapRow(ResultSet resultSet, int rowNum) throws SQLException {
                    String url = resultSet.getString(FIELD_URL);
                    Long hostId = resultSet.getLong(FIELD_HOST_ID);
                    Date receiveTime = resultSet.getDate(FIELD_RECEIVE_TIME);
                    return new XssNotificationInfo(url, hostId, receiveTime);
                }
            };

    public Long addXssNotification(final String url, final Long hostId, final Date date) throws InternalException {
        GeneratedKeyHolder generatedKeyHolder = new GeneratedKeyHolder();
        PreparedStatementCreator psc = new PreparedStatementCreator() {
            @Override
            public PreparedStatement createPreparedStatement(Connection connection) throws SQLException {
                PreparedStatement ps = connection.prepareStatement(INSERT_XSS_NOTIFICATION_QUERY,
                        Statement.RETURN_GENERATED_KEYS);
                ps.setString(1, url);
                ps.setLong(2, hostId);
                ps.setTimestamp(3, new Timestamp(date.getTime()));
                return ps;
            }
        };

        getJdbcTemplate(WMCPartition.nullPartition()).getJdbcOperations().update(psc, generatedKeyHolder);
        return generatedKeyHolder.getKey().longValue();
    }

    public XssNotificationInfo getXssNotification(Long xssNotificationId) throws InternalException {
        return getJdbcTemplate(WMCPartition.nullPartition()).queryForObject(
                SELECT_XSS_QUERY,
                XSS_NOTIFICATION_MAPPER,
                xssNotificationId
        );
    }

    @Override
    public XssNotificationInfo getSingleNotification(Long id) throws InternalException {
        return getXssNotification(id);
    }
}
