package ru.yandex.webmaster.periodic.host;

import org.joda.time.DateTime;
import org.joda.time.Months;
import org.joda.time.ReadablePeriod;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.webmaster.common.host.HostEvent;
import ru.yandex.webmaster.common.host.HostEventType;
import ru.yandex.webmaster.common.host.dao.TblHostsMainDao;
import ru.yandex.wmconsole.data.AddUrlRequest;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.HostInfo;
import ru.yandex.wmconsole.service.AddUrlService;
import ru.yandex.wmconsole.service.HostInfoService;
import ru.yandex.wmconsole.service.error.WMCUserProblem;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.servantlet.AbstractServantlet;

/**
 * @author aherman
 */
public class AddUrlForVerifiedHostTask extends AbstractHostEventTask {
    private static final Logger log = LoggerFactory.getLogger(AddUrlForVerifiedHostTask.class);
    private static final int NEW_EVENTS_BATCH_SIZE = 1024;

    private HostInfoService hostInfoService;
    private AddUrlService addUrlService;
    private TblHostsMainDao tblHostsMainDao;

    private ReadablePeriod addUrlPeriod = Months.ONE;

    @Override
    protected void process(HostEvent hostEvent) throws InternalException, UserException {
        log.info("AddUrl for verified host: " + hostEvent);
        DateTime hostInfoStatusDate = null;

        BriefHostInfo briefHostInfo = tblHostsMainDao.getBriefHostInfoByHostId(hostEvent.getHostId());
        if (briefHostInfo == null) {
            log.warn("Host not found: event=" + hostEvent);
            return;
        }

        HostInfo hostInfo = hostInfoService.getHostInfo(hostEvent.getEventForUserId(), briefHostInfo);
        if (hostInfo != null && hostInfo.getHostInfoStatusDate() != null) {
            hostInfoStatusDate = new DateTime(hostInfo.getHostInfoStatusDate());
        }

        DateTime olderThanDate = DateTime.now().minus(addUrlPeriod);
        if (hostInfoStatusDate != null && olderThanDate.isBefore(hostInfoStatusDate)) {
            log.info("Host info too yong: hostInfoStatusDate=" + hostInfoStatusDate);
            return;
        }

        AddUrlRequest addUrlRequest = new AddUrlRequest(AbstractServantlet.prepareUrl(briefHostInfo.getName(), false), null,
                hostEvent.getEventByUserId(), null, null);
        try {
            addUrlService.addUrl(addUrlRequest, false, "ru", false);
        } catch (UserException e) {
            if (e.getProblem() == WMCUserProblem.URL_IS_ALREADY_INDEXED) {
                log.warn("Host already indexed: event=" + hostEvent, e);
            } else {
                throw new InternalException(InternalProblem.PROCESSING_ERROR, "Unable to addUrl: event=" + hostEvent, e);
            }
        }
        log.info("AddUrl for verified host complete: event=" + hostEvent);
    }

    @Override
    protected int getRetryCount() {
        return 2;
    }

    @Override
    protected int getBatchSize() {
        return NEW_EVENTS_BATCH_SIZE;
    }

    @Override
    protected HostEventTaskType getTaskType() {
        return HostEventTaskType.VERIFIED_HOST_ADD_URL;
    }

    @Override
    protected HostEventType getEventType() {
        return HostEventType.HOST_VERIFIED;
    }

    @Required
    public void setAddUrlService(AddUrlService addUrlService) {
        this.addUrlService = addUrlService;
    }

    @Required
    public void setHostInfoService(HostInfoService hostInfoService) {
        this.hostInfoService = hostInfoService;
    }

    @Required
    public void setTblHostsMainDao(TblHostsMainDao tblHostsMainDao) {
        this.tblHostsMainDao = tblHostsMainDao;
    }
}
