package ru.yandex.webmaster.periodic.host.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import org.joda.time.DateTime;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.transaction.TransactionStatus;

import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.db.q.SqlOrder;
import ru.yandex.webmaster.periodic.host.HostEventTaskType;
import ru.yandex.webmaster.periodic.host.LastEventId;
import ru.yandex.webmaster.periodic.host.LastEventState;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.util.ServiceTransactionCallbackWithoutResult;

/**
 * CREATE TABLE IF NOT EXISTS tbl_host_event_task_state (
 *     task_id SMALLINT NOT NULL
 *   , event_id INT NOT NULL
 *   , date TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
 *   , retry_count SMALLINT NOT NULL
 *   , state SMALLINT NOT NULL
 *
 *   , PRIMARY KEY (task_id, event_id, state)
 * ) ENGINE=InnoDB;
 *
 * @author aherman
 */
public class TblHostEventTaskStateDao extends AbstractDbService {
    static final String COLUMN_TASK_ID = "task_id";
    static final String COLUMN_EVENT_ID = "event_id";
    static final String COLUMN_DATE = "date";
    static final String COLUMN_RETRY_COUNT = "retry_count";
    static final String COLUMN_STATE = "state";

    public List<LastEventId> getEvents(HostEventTaskType taskType, SqlCondition c, SqlLimits limits, SqlOrder order)
            throws InternalException
    {
        c = TblHostEventTaskStateCondition.task(taskType).and(c);
        String q = "SELECT * FROM tbl_host_event_task_state WHERE " + c.sql() + " " + order.toSql() + " " + limits.toMysqlLimits();
        return getJdbcTemplate(WMCPartition.nullPartition()).query(q, getLastEventMapper(), c.args().toArray());
    }

    public void remove(LastEventId lastEventId) throws InternalException {
        String q = "DELETE FROM tbl_host_event_task_state WHERE task_id = ? AND event_id = ? AND state = ?";
        getJdbcTemplate(WMCPartition.nullPartition())
                .update(q, lastEventId.getTaskType().value(), lastEventId.getEventId(), lastEventId.getState().value());
    }

    public void replaceEvent(LastEventId lastEventId) throws InternalException {
        String q = "REPLACE INTO tbl_host_event_task_state(task_id, event_id, state, retry_count) VALUES (?, ?, ?, ?)";
        getJdbcTemplate(WMCPartition.nullPartition())
                .update(q, lastEventId.getTaskType().value(), lastEventId.getEventId(), lastEventId.getState().value(),
                        lastEventId.getRetryCount());
    }

    public void replaceEvent(final LastEventId oldEvent, final LastEventId newEvent) throws UserException,
            InternalException
    {
        getServiceTransactionTemplate(WMCPartition.nullPartition()).executeInService(
                new ServiceTransactionCallbackWithoutResult() {
                    @Override
                    protected void doInTransactionWithoutResult(TransactionStatus transactionStatus) throws
                            UserException,
                            InternalException
                    {
                        if (oldEvent != null) {
                            remove(oldEvent);
                        }
                        replaceEvent(newEvent);
                    }
                });
    }

    private static ParameterizedRowMapper<LastEventId> getLastEventMapper() {
        return new ParameterizedRowMapper<LastEventId>() {
            @Override
            public LastEventId mapRow(ResultSet rs, int rowNum) throws SQLException {
                HostEventTaskType taskType = HostEventTaskType.R.fromValueOrUnknown(rs.getInt(COLUMN_TASK_ID));
                int eventId = rs.getInt(COLUMN_EVENT_ID);
                DateTime date = new DateTime(rs.getTimestamp(COLUMN_DATE));
                int retryCount = rs.getInt(COLUMN_RETRY_COUNT);
                LastEventState state = LastEventState.R.fromValueOrUnknown(rs.getInt(COLUMN_STATE));
                return new LastEventId(taskType, eventId, retryCount, state, date);
            }
        };
    }

    public static SqlOrder orderByEventIdDesc() {
        return SqlOrder.orderByColumnDesc(COLUMN_EVENT_ID);
    }
}
