package ru.yandex.webmaster.periodic.service;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.http.HttpEntity;
import org.apache.http.HttpRequest;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.HostDbHostInfo;
import ru.yandex.wmconsole.util.BackportedIdUtils;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.sita.UserAgentEnum;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author avhaliullin
 */
public class NewWebmasterCoordinatorProxyService {
    private static final ObjectMapper OM = new ObjectMapper();
    private static final String PATH_BASE = "/old-webmaster/api/sitemaps/";
    private static final DateFormat DATE_TIME_FORMAT = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");

    private final CloseableHttpClient httpClient;

    private String webmasterCoordinatorUrl;

    public NewWebmasterCoordinatorProxyService() {
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectionRequestTimeout(2000)
                .setConnectTimeout(500)
                .setSocketTimeout(5000)
                .build();

        httpClient = HttpClients.custom()
                .setMaxConnTotal(16)
                .setMaxConnPerRoute(16)
                .setConnectionTimeToLive(10, TimeUnit.SECONDS)
                .setDefaultRequestConfig(requestConfig)
                .setUserAgent(UserAgentEnum.WEBMASTER.getValue())
                .build();
    }

    public List<SitemapInfo> getSitemaps(HostDbHostInfo hostInfo) throws InternalException {
        HttpGet request = new HttpGet(webmasterCoordinatorUrl + PATH_BASE + "list.json?hostId=" + hostInfo2HostIdString(hostInfo));

        WebmasterResponse response = doRequest(request);
        if (!response.errors.isEmpty()) {
            throw errorResponse(response.errors);
        }
        if (response.data == null || !(response.data instanceof ObjectNode)) {
            throw unexpectedResponse(request, "data expected to be object, but found: " + String.valueOf(response.data));
        }
        ObjectNode dataObject = (ObjectNode) response.data;
        JsonNode sitemapsArrayNode = dataObject.get("sitemaps");
        List<SitemapInfo> result = new ArrayList<>();
        for (JsonNode sitemapNode : sitemapsArrayNode) {
            Date addDate = getDate(request, sitemapNode, "addDate");
            Date deleteDate = getDate(request, sitemapNode, "deleteDate");
            long userId = Long.parseLong(sitemapNode.get("addedByUserId").asText());
            String sitemapId = sitemapNode.get("sitemapId").asText();
            String url = sitemapNode.get("sitemapUrl").asText();
            boolean deleted = sitemapNode.get("deleted").asBoolean();
            Date modificationDate = deleted ? deleteDate : addDate;
            result.add(new SitemapInfo(userId, sitemapId, url, modificationDate, deleted));
        }
        return result;
    }

    public void addSitemap(HostDbHostInfo hostInfo, long userId, String url) throws InternalException {
        HttpGet request = new HttpGet(webmasterCoordinatorUrl + PATH_BASE + "add.json?" +
                "hostId=" + hostInfo2HostIdString(hostInfo) +
                "&userId=" + userId +
                "&sitemapUrl=" + urlencode(url));
        WebmasterResponse response = doRequest(request);
        if (!response.errors.isEmpty()) {
            throw errorResponse(response.errors);
        }
    }

    public void removeSitemap(HostDbHostInfo hostInfo, String sitemapId) throws InternalException {
        HttpGet request = new HttpGet(webmasterCoordinatorUrl + PATH_BASE + "remove.json?" +
                "hostId=" + hostInfo2HostIdString(hostInfo) +
                "&sitemapId=" + sitemapId);
        WebmasterResponse response = doRequest(request);
        if (!response.errors.isEmpty()) {
            throw errorResponse(response.errors);
        }
    }

    private Date getDate(HttpUriRequest request, JsonNode object, String fieldName) throws InternalException {
        if (!object.has(fieldName)) {
            return null;
        }
        JsonNode node = object.get(fieldName);
        if (node.isNull()) {
            return null;
        }
        try {
            return DATE_TIME_FORMAT.parse(node.asText());
        } catch (ParseException e) {
            throw unexpectedResponse(request, "Date in unexpected format: " + node.asText(), e);
        }
    }

    private InternalException errorResponse(List<NewWebmasterError> errors) {
        StringBuilder sb = new StringBuilder();
        for (NewWebmasterError error : errors) {
            sb.append("{code: ").append(error.code).append(", message: ").append(String.valueOf(error.message)).append("};");
        }
        return new InternalException(InternalProblem.PROCESSING_ERROR, "New webmaster backend responded with error: " + sb.toString());
    }

    private WebmasterResponse doRequest(HttpUriRequest request) throws InternalException {
        try (CloseableHttpResponse response = httpClient.execute(request)) {
            HttpEntity entity = response.getEntity();
            if (entity == null) {
                throw unexpectedResponse(request, "Response entity not found");
            }
            JsonNode root = OM.readTree(entity.getContent());
            if (!(root instanceof ObjectNode)) {
                throw unexpectedResponse(request, "Root node expected to be object, but found " + root);
            }
            ObjectNode rootObject = (ObjectNode) root;
            JsonNode errors = rootObject.get("errors");

            List<NewWebmasterError> errorsList = new ArrayList<>();
            if (errors != null && errors.isArray()) {
                for (JsonNode errNode : errors) {
                    String code = errNode.get("code").textValue();
                    JsonNode messageNode = errNode.get("message");
                    String message = null;
                    if (messageNode != null && !messageNode.isNull()) {
                        message = messageNode.textValue();
                    }
                    errorsList.add(new NewWebmasterError(code, message));
                }
            }
            return new WebmasterResponse(errorsList, rootObject.get("data"));
        } catch (IOException e) {
            throw new InternalException(InternalProblem.CONNECTION_PROBLEM,
                    "New webmaster backend request failed: " + request.toString(), e);
        }
    }

    public static class SitemapInfo {
        public final long userId;
        public final String id;
        public final String url;
        public final Date modificationDate;
        public final boolean deleted;

        public SitemapInfo(long userId, String id, String url, Date modificationDate, boolean deleted) {
            this.userId = userId;
            this.id = id;
            this.url = url;
            this.modificationDate = modificationDate;
            this.deleted = deleted;
        }
    }

    private static class WebmasterResponse {
        public final List<NewWebmasterError> errors;
        public final JsonNode data;

        public WebmasterResponse(List<NewWebmasterError> errors, JsonNode data) {
            this.errors = errors;
            this.data = data;
        }
    }

    private static class NewWebmasterError {
        public final String code;
        public final String message;

        public NewWebmasterError(String code, String message) {
            this.code = code;
            this.message = message;
        }
    }

    private InternalException unexpectedResponse(HttpRequest request, String message, Throwable t) {
        return new InternalException(InternalProblem.PROCESSING_ERROR, "Unexpected webmaster response for request " + request.toString() + ". " + message, t);
    }

    private InternalException unexpectedResponse(HttpRequest request, String message) {
        return unexpectedResponse(request, message, null);
    }

    private String urlencode(String s) {
        try {
            return URLEncoder.encode(s, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException("Should never happen", e);
        }
    }

    private String hostInfo2HostIdString(HostDbHostInfo hostInfo) throws InternalException {
        return urlencode(BackportedIdUtils.urlToHostId(hostInfo.getName()).toStringId());
    }

    @Required
    public void setWebmasterCoordinatorUrl(String webmasterCoordinatorUrl) {
        this.webmasterCoordinatorUrl = webmasterCoordinatorUrl;
    }

}
