package ru.yandex.wmconsole.notifier.sender;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.text.DateFormat;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;

import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.transform.JDOMSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.xml.sax.InputSource;

import ru.yandex.wmconsole.data.LanguageEnum;
import ru.yandex.wmconsole.data.NotificationChannelEnum;
import ru.yandex.wmconsole.data.NotificationTypeEnum;
import ru.yandex.wmconsole.data.NotificationsForUser;
import ru.yandex.wmconsole.data.info.NotificationInfo;
import ru.yandex.wmconsole.notifier.provider.Provider;
import ru.yandex.wmconsole.service.IWMCUserInfoService;
import ru.yandex.wmtools.common.data.info.WMUserInfo;
import ru.yandex.wmtools.common.data.wrappers.UserInfoWrapper;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.error.UserException;

/**
 * @author Andrey Mima (amima@yandex-team.ru)
 */
public abstract class AbstractSender implements Sender {
    private static final Logger log = LoggerFactory.getLogger(AbstractSender.class);

    protected static final String CHANNEL_ATTRIBUTE = "channel";
    private static final String TAG_NOTIFICATIONS = "notifications";
    private static final String TAG_NOTIFICATION = "notification";
    private static final String TAG_NOTIFICATION_ID = "id";
    private static final String TAG_NOTIFICATION_USERID = "userid";
    private static final String TAG_NOTIFICATION_ISSUEID = "issueid";
    private static final String TAG_NOTIFICATION_HAPPENED = "happened";
    private static final String TAG_NOTIFICATION_TYPE = "type";
    private static final String TAG_USER_INFO = "emailuserinfo";

    protected IWMCUserInfoService userInfoService;

    private File senderXslFile;
    private File senderNoxslFile;

    private File ruNotificationXslFile;
    private File ukNotificationXslFile;
    private File enNotificationXslFile;
    private File trNotificationXslFile;

    protected final Map<NotificationTypeEnum, Provider> providers =
            new EnumMap<NotificationTypeEnum, Provider>(NotificationTypeEnum.class);

    protected Document getNotificationDocument(NotificationsForUser notificationsForUser, NotificationChannelEnum channel)
            throws InternalException, UserException {
        Document document = new Document();

        List<NotificationInfo> notifications = notificationsForUser.getAllNotifications();
        if (notifications == null || notifications.isEmpty()) {
            log.error("Notifications to send null or zero size!");
            throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "Notifications to send more than one");
        }
        Element rootElement = new Element(TAG_NOTIFICATIONS);
        rootElement.setAttribute(TAG_NOTIFICATION_TYPE, notifications.get(0).getNotificationType().toString());//new!!!

        for (NotificationInfo notification : notifications) {
            List<Element> notificationElements = notification.getNotificationType() < 1024 ?
                    providers.get(
                            NotificationTypeEnum.R.fromValueOrNull(notification.getNotificationType())).
                            getNotificationElements(notification, channel)
                    :
                    providers.get(NotificationTypeEnum.HOST_OWNERS).
                            getNotificationElements(notification, channel);
            if (notificationElements == null || notificationElements.isEmpty()) {
                throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "Empty notification elements from provider");
            }
            Element notificationElement = getNotificationElement(notification);
            notificationElement.setContent(notificationElements);
            rootElement.addContent(notificationElement);
        }

        document.setRootElement(rootElement);

        // TAG_USER_INFO used in webmaster.email.hello tanker template
        if (channel != NotificationChannelEnum.MESSAGE) {
            try {
                WMUserInfo userInfo = userInfoService.getUserInfo(notificationsForUser.getUserId());
                StringBuilder sb = new StringBuilder();
                new UserInfoWrapper(userInfo).toXml(sb);
                Element userInfoElement = new Element(TAG_USER_INFO);
                SAXBuilder b = new SAXBuilder();
                Element userElement = b.build(new InputSource(new StringReader(sb.toString()))).getRootElement();
                userElement.detach();
                userInfoElement.addContent(userElement);
                document.getRootElement().addContent(userInfoElement);
            } catch (JDOMException e) {
                log.error("Invalid user wrapper", e);
            } catch (IOException e) {
                log.error("Invalid user wrapper", e);
            }
        }

        return document;
    }

    private Element getNotificationElement(NotificationInfo notification) {
        Element notificationElement = new Element(TAG_NOTIFICATION);
        notificationElement.setAttribute(TAG_NOTIFICATION_ID, notification.getNotificationId().toString());
        notificationElement.setAttribute(TAG_NOTIFICATION_USERID, notification.getUserId().toString());
        notificationElement.setAttribute(TAG_NOTIFICATION_ISSUEID, notification.getIssueId().toString());
        notificationElement.setAttribute(TAG_NOTIFICATION_HAPPENED,
                DateFormat.getDateTimeInstance().format(notification.getHappenedTime()));
        notificationElement.setAttribute(TAG_NOTIFICATION_TYPE, notification.getNotificationType().toString());//new!!!
        return notificationElement;
    }

    protected String extractXmlString(NotificationsForUser notificationsForUser, NotificationChannelEnum channel,
                                      boolean isXsltNeeded)
            throws TransformerException, UserException, InternalException {
        File xslFile = isXsltNeeded ? senderXslFile : senderNoxslFile;
        Document document = getNotificationDocument(notificationsForUser, channel);
        document.getRootElement().setAttribute(CHANNEL_ATTRIBUTE, Integer.toString(channel.getValue()));
        Source srcXml = new JDOMSource(document);
        TransformerFactory xslTransformerFactory = TransformerFactory.newInstance();
        //TODO extract file stream
        Source xslSource = new StreamSource(xslFile);
        Transformer transformer = xslTransformerFactory.newTransformer(xslSource);
        Writer writer = new StringWriter();
        transformer.transform(srcXml, new StreamResult(writer));
        return writer.toString();
    }

    public String extractXmlString(
            NotificationsForUser notificationsForUser,
            NotificationChannelEnum channel,
            LanguageEnum lang,
            boolean isXsltNeeded) throws UserException, InternalException, TransformerException {
        if (lang == null) {
            log.debug("setting email notification language to default " + LanguageEnum.DEFAULT_EMAIL_LANGUAGE);
            lang = LanguageEnum.DEFAULT_EMAIL_LANGUAGE;
        }
        File xslFile;
        if (isXsltNeeded) {
            switch(lang) {
                case RU: xslFile = ruNotificationXslFile; break;
                case UK: xslFile = ukNotificationXslFile; break;
                case EN: xslFile = enNotificationXslFile; break;
                case TR: xslFile = trNotificationXslFile; break;
                default:
                    throw new InternalException(InternalProblem.ILLEGAL_ARGUMENT,
                            "Unable to find email template for lang: " + lang);
            }
        } else {
            xslFile = senderNoxslFile;
        }
        Document document = getNotificationDocument(notificationsForUser, channel);
        document.getRootElement().setAttribute(CHANNEL_ATTRIBUTE, Integer.toString(channel.getValue()));
        Source srcXml = new JDOMSource(document);
        TransformerFactory xslTransformerFactory = TransformerFactory.newInstance();
        Source xslSource = new StreamSource(xslFile);
        Transformer transformer = xslTransformerFactory.newTransformer(xslSource);
        Writer writer = new StringWriter();
        transformer.transform(srcXml, new StreamResult(writer));
        return writer.toString();
    }

    public void setProviders(Map<NotificationTypeEnum, Provider> providers) {
        this.providers.putAll(providers);
    }

    @Required
    public void setUserInfoService(IWMCUserInfoService userInfoService) {
        this.userInfoService = userInfoService;
    }

    @Required
    public void setSenderXslFile(File senderXslFile) {
        this.senderXslFile = senderXslFile;
    }

    @Required
    public void setSenderNoxslFile(File senderNoxslFile) {
        this.senderNoxslFile = senderNoxslFile;
    }

    @Required
    public void setRuNotificationXslFile(File ruNotificationXslFile) {
        this.ruNotificationXslFile = ruNotificationXslFile;
    }

    @Required
    public void setUkNotificationXslFile(File ukNotificationXslFile) {
        this.ukNotificationXslFile = ukNotificationXslFile;
    }

    @Required
    public void setEnNotificationXslFile(File enNotificationXslFile) {
        this.enNotificationXslFile = enNotificationXslFile;
    }

    @Required
    public void setTrNotificationXslFile(File trNotificationXslFile) {
        this.trNotificationXslFile = trNotificationXslFile;
    }
}
