package ru.yandex.wmconsole.periodic;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.scheduler.ExecutionContext;
import ru.yandex.webmaster.common.host.dao.TblHostsMainDao;
import ru.yandex.wmconsole.data.NotificationTypeEnum;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.HostDailyStateInfo;
import ru.yandex.wmconsole.data.info.UsersHostsInfo;
import ru.yandex.wmconsole.service.CollectErrorsGrowService;
import ru.yandex.wmconsole.service.ErrorsGrowService;
import ru.yandex.wmconsole.service.HostDailyStateService;
import ru.yandex.wmconsole.service.HostInfoService;
import ru.yandex.wmconsole.service.NotificationService;
import ru.yandex.wmconsole.service.UserOptionsService;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.util.scheduler.timetable.AbstractTaskExecutor;

/**
 * @author Andrey Mima (amima@yandex-team.ru)
 */
public class ErrorsGrowCollectorTask extends AbstractTaskExecutor{
    private static final Logger log = LoggerFactory.getLogger(ErrorsGrowCollectorTask.class);

    private static final int MIN_ERRORS = 10;

    private NotificationService notificationService;
    private ErrorsGrowService errorsGrowService;
    private UserOptionsService userOptionsService;
    private HostDailyStateService hostDailyStateService;
    private CollectErrorsGrowService collectErrorsGrowService;
    private HostInfoService hostInfoService;
    private TblHostsMainDao tblHostsMainDao;

    private Long fillTables(Long hostId, Integer percent, Date date) throws InternalException {
        return errorsGrowService.addErrorsGrowNotification(hostId, percent, date);
    }

    private Map<Long, Set<Long>> getHostUsersMap() throws UserException, InternalException {
        Map<Long, Set<Long>> result = new HashMap<Long, Set<Long>>(1 << 16);
        List<UsersHostsInfo> usersHostsInfoList = collectErrorsGrowService.getUnprocessedHosts();
        for (UsersHostsInfo info : usersHostsInfoList) {
            if (!info.getVerificationState().isVerified()) {
                continue;
            }
            Long hostId = info.getHostId();
            if (!result.containsKey(hostId)) {
                result.put(hostId, new HashSet<Long>());
            }
            Long userId = info.getUserId();
            Set<Long> hostUsers = result.get(hostId);
            if (!hostUsers.contains(userId)) {
                hostUsers.add(userId);
            }
        }
        return result;
    }

    @Override
    public String runWithRELogging(ExecutionContext context) {
        String msg = getClass().getName() + " task executed";
        logUserDbConnections();
        try {
            errorsGrowService.refreshConcurrentHosts();

            Map<Long, Set<Long>> hostUsersMap = getHostUsersMap();
            Map<Long, Integer> hostPercents = new HashMap<Long, Integer>(1 << 10);

            Set<Long> toRemoveHosts = new HashSet<Long>();
            for (Long hostId : hostUsersMap.keySet()) {
                BriefHostInfo hostInfo = tblHostsMainDao.getBriefHostInfoByHostId(hostId);

                Set<Long> toRemove = new HashSet<Long>();
                Set<Long> userIds = hostUsersMap.get(hostId);
                for (Long userId : userIds) {
                    try {
                        Integer userPercent = userOptionsService.getUserOptions(userId).getErrorsGrow();

                        List<HostDailyStateInfo> allStates =
                                hostDailyStateService.getHostDailyState(userId, hostInfo);

                        HostDailyStateInfo currentState;
                        HostDailyStateInfo lastState;

                        if (allStates.size() < 3) {
                            log.debug("states quantity less than 3 for host " + hostInfo.getName());
                            toRemove.add(userId);
                            continue;
                        } else {
                            currentState = allStates.listIterator(allStates.size() - 1).previous();
                            lastState = allStates.listIterator(allStates.size() - 2).previous();
                        }

                        Integer hostPercent;
                        if (lastState.getSiteErrors() >= MIN_ERRORS) {
                            hostPercent =
                                    (int) ((((float) currentState.getSiteErrors() / (float) lastState.getSiteErrors()) - 1) * 100);
                        } else {
                            hostPercent = Integer.MIN_VALUE;
                        }
                        if (hostPercent >= userPercent) {
                            hostPercents.put(hostId, userPercent);
                            log.debug("Grow host: " + hostId.toString() +
                                    " Percent: " + hostPercent +
                                    " Last: " + lastState.getSiteErrors() +
                                    " (" + lastState.getStateDate().toString() + ")" +
                                    " Curr: " + currentState.getSiteErrors() +
                                    " (" + currentState.getStateDate().toString() + ")");
                        } else {
                            toRemove.add(userId);
                        }
                    } catch (NoSuchElementException e) {
                        log.warn("no previous state for host " + hostId);
                    }
                }

                userIds.removeAll(toRemove);

                if (userIds.size() == 0) {
                    toRemoveHosts.add(hostId);
                }
            }

            for (Long hostId : toRemoveHosts) {
                hostUsersMap.remove(hostId);
            }

            Date date = new Date();
            for (Long hostId : hostPercents.keySet()) {
                Integer percent = hostPercents.get(hostId);
                Long issueId = fillTables(hostId, percent, date);
                List<Long> userIds = new ArrayList<Long>(hostUsersMap.get(hostId));
                if (userIds.size() > 0) {
                    notificationService.insertNotificationForUsers(NotificationTypeEnum.ERRORS_GROW, issueId, userIds, date);
                }
            }
        } catch (ClassCastException e) {
            msg = "ClassCastException in " + getClass().getName() + " " +
                    "while extracting notification data (possible corrupt xml)";
            log.error(msg, e);
        } catch (InternalException e) {
            msg = "InternalException in " + getClass().getName() + " " +
                    "while inserting notification(s)";
            log.error(msg, e);
        } catch (UserException e) {
            msg = "UserException in " + getClass().getName() + " " +
                    "while inserting notification(s)";
            log.error(msg, e);
        }
        logUserDbConnections();
        return msg;
    }

    @Required
    public void setNotificationService(NotificationService notificationService) {
        this.notificationService = notificationService;
    }

    @Required
    public void setErrorsGrowService(ErrorsGrowService errorsGrowService) {
        this.errorsGrowService = errorsGrowService;
    }

    @Required
    public void setUserOptionsService(UserOptionsService userOptionsService) {
        this.userOptionsService = userOptionsService;
    }

    @Required
    public void setHostDailyStateService(HostDailyStateService hostDailyStateService) {
        this.hostDailyStateService = hostDailyStateService;
    }

    @Required
    public void setCollectErrorsGrowService(CollectErrorsGrowService collectErrorsGrowService) {
        this.collectErrorsGrowService = collectErrorsGrowService;
    }

    @Required
    public void setHostInfoService(HostInfoService hostInfoService) {
        this.hostInfoService = hostInfoService;
    }

    @Required
    public void setTblHostsMainDao(TblHostsMainDao tblHostsMainDao) {
        this.tblHostsMainDao = tblHostsMainDao;
    }
}
