package ru.yandex.wmconsole.periodic;

import java.util.Calendar;
import java.util.Date;
import java.util.EnumSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.scheduler.ExecutionContext;
import ru.yandex.wmconsole.data.HostInfoStatusEnum;
import ru.yandex.wmconsole.data.NotificationTypeEnum;
import ru.yandex.wmconsole.data.info.HostStatusInfo;
import ru.yandex.wmconsole.data.info.HostUnavailableInfo;
import ru.yandex.wmconsole.data.info.UsersHostsInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmconsole.service.HostInfoService;
import ru.yandex.wmconsole.service.HostStatusService;
import ru.yandex.wmconsole.service.SendInternalNotificationService;
import ru.yandex.wmconsole.service.UsersHostsService;
import ru.yandex.wmtools.common.util.scheduler.timetable.AbstractLockableTaskExecutor;

/**
 * Периодическая задача для отправки сообщений о недоступности сайта
 *
 * User: Alexey Zakharov <azakharov@yandex-team.ru>
 * Date: 29.12.11
 */
public class HostUnavailabilityNotificationTask extends AbstractLockableTaskExecutor {
    private static final Logger log = LoggerFactory.getLogger(HostUnavailabilityNotificationTask.class);

    private static final String PARAM_HOST_ID = "host_id";
    private static final String PARAM_HOST_STATUS = "status";
    private static final String PARAM_SPIDER_IP = "spider_ip";
    private static final String PARAM_LAST_ACCESS = "last_access";

    private static final String HOST_STATUS_LOCK = "host_status_lock";

    private HostInfoService hostInfoService;
    private HostStatusService hostStatusService;
    public UsersHostsService usersHostsService;
    private SendInternalNotificationService sendInternalNotificationService;

    private static final EnumSet<HostInfoStatusEnum> BAD_STATUS_VALUES = EnumSet.of(
            HostInfoStatusEnum.WARN_CONNECTION_FAILED,
            HostInfoStatusEnum.WARN_DISALLOW,

            HostInfoStatusEnum.WILL_CONNECTION_FAILED,
            HostInfoStatusEnum.WILL_DNS_ERROR,
            HostInfoStatusEnum.WILL_DISALLOW
    );


    @Override
    public String runWithRELogging(final ExecutionContext context) {
        log.debug("Host unavalability notifier triggered");
        try {
            if (!getLock(WMCPartition.nullPartition(), HOST_STATUS_LOCK, 2 * 24 * 60)) {
                String msg = "Failed to get lock for host unavailability notification task";
                log.warn(msg);
                return msg;
            }

            long ms1 = Calendar.getInstance().getTimeInMillis();
            logUserDbConnections();
            // getHostDbCount возвращает количество хостовых баз (host0 - host7)
            // Всего различных баз 9 штук (8 хостовых и 1 пользовательская), поэтому возвращается wholeCount - 1
            final int  dbCount = WMCPartition.getHostDbCount(getDatabaseCount());

            for (int dbIndex = 0; dbIndex < dbCount; dbIndex++) {
                logConnections(dbIndex);
                // Получаем хосты с ошибочным статусом из tbl_penalty_info
                final List<HostUnavailableInfo> statusInfos = hostInfoService.getBadPenaltyHosts(dbIndex);
                // Создаем список для хранения идентификтаоров хостов, записи для которых нужно оставить в БД,
                // чтобы не отправлять повторно нотификации для одной и той же ошибки, если статус не изменился
                final List<Long> hostsToHold = new LinkedList<Long>();
                // Получаем хосты с изменившимся статусом из БД "host" + dbIndex
                final List<HostUnavailableInfo> newBadHosts = hostStatusService.getChangedHosts(statusInfos, dbIndex, hostsToHold);
                log.debug("hostsToHold.size = " + hostsToHold.size());

                // Создаем запрос для отправки нотификации
                for (HostUnavailableInfo hsi : newBadHosts ) {
                    final long hostId = hsi.getHostId();
                    final HostInfoStatusEnum status = hsi.getCalculatedHostInfoStatus();
                    if (!BAD_STATUS_VALUES.contains(status)) {
                        continue;
                    }

                    final String hostName = hsi.getHostName();
                    try {
                        final Long hostIdFromUserDb = hostInfoService.getHostIdByHostName(hostName, true);
                        if (hostIdFromUserDb == null) {
                            log.debug("Host " + hostName + " not found in user_db");
                            continue;
                        }

                        List<UsersHostsInfo> users = usersHostsService.listVerifiedUsersForHost(hostIdFromUserDb);
                        if (users.size() == 0) {
                            // не отправляем сообщение по неподтвержденным сайтам
                            continue;
                        }

                        final String spiderIp = hsi.getSpiderIp();
                        final Date lastAccess = hsi.getLastAccess();
                        if (lastAccess == null) {
                            log.error("last_access is null for dbindex=" + dbIndex + " host_id=" + hostId);
                            continue;
                        }
                        final Date lastAccessYa = hsi.getLastAccessYa();
                        final Date lastAccessHtarc = hsi.getLastAccessHtarc();

                        if (lastAccessYa == null ||
                                lastAccessHtarc == null ||
                                lastAccessYa != null && lastAccessHtarc != null && lastAccessYa.before(lastAccessHtarc)) {
                            // рассинхронизированное состояние:
                            // не загружали из обоих источников или для хтарка данные свежее, чем для ya
                            log.error("last_access_ya < last_access_htarc for dbindex=" + dbIndex +
                                    " host_id=" + hostId +
                                    " lastAccessYa=" + lastAccessYa + " lastAccessHtarc=" + lastAccessHtarc);
                            continue;
                        }
                        final Map<String, Object> params = getParams(hostIdFromUserDb, status.getValue(), spiderIp, lastAccess);
                        HostStatusInfo statusInfo = hsi.getStatus();
                        log.debug("Sending notification " + hostName + " " + hostId + " " +
                                statusInfo.getHostInfoStatusHtarc() + " " + statusInfo.getPenaltyHtarc() + " " +
                                statusInfo.getHostInfoStatusYa() + " " + statusInfo.getPenaltyYa() + " " +
                                hsi.getSpiderIp() + " " + hsi.getLastAccess());
                        sendInternalNotificationService.sendInternalNotification(
                                NotificationTypeEnum.HOST_STATUS_CHANGED, params);
                        hostStatusService.saveBadHost(dbIndex, hostId, status.getValue());
                    } catch (Exception e) {
                        log.error("Error while sending notification HOST_STATUS_CHANGED, host_id = " + hostId +
                                ",  status = " + status, e);
                    }
                }
                hostStatusService.saveBadHosts(newBadHosts, dbIndex, hostsToHold);
                logConnections(dbIndex);
            }
            long ms2 = Calendar.getInstance().getTimeInMillis();
            logUserDbConnections();
            log.info("Processing time is " + (ms2 - ms1));
        } catch (Exception e) {
            log.error("Exception in HostUnavailabilityNotificationTask", e);
        } finally {
            releaseLock(WMCPartition.nullPartition(), HOST_STATUS_LOCK);
        }
        return "HostUnavailabilityNotificationTask task executed";
    }

    private Map<String, Object> getParams(final long hostId, final int hostStatus, final String spiderIp, final Date lastAccess) {
        final Map<String, Object> result = new LinkedHashMap<String, Object>();
        result.put(PARAM_HOST_ID, hostId);
        result.put(PARAM_HOST_STATUS, hostStatus);
        if (spiderIp != null) {
            result.put(PARAM_SPIDER_IP, spiderIp);
        }
        if (lastAccess != null) {
            result.put(PARAM_LAST_ACCESS, lastAccess);
        }
        log.debug("Sending notification with params:");
        for (Map.Entry<String, Object> p : result.entrySet()) {
            log.debug(p.getKey() + "=" + p.getValue());
        }
        return result;
    }

    @Required
    public void setHostInfoService(final HostInfoService hostInfoService) {
        this.hostInfoService = hostInfoService;
    }

    @Required
    public void setHostStatusService(final HostStatusService hostStatusService) {
        this.hostStatusService = hostStatusService;
    }

    @Required
    public void setSendInternalNotificationService(final SendInternalNotificationService sendInternalNotificationService) {
        this.sendInternalNotificationService = sendInternalNotificationService;
    }

    @Required
    public void setUsersHostsService(UsersHostsService usersHostsService) {
        this.usersHostsService = usersHostsService;
    }
}
