package ru.yandex.wmconsole.periodic;

import java.util.concurrent.TimeUnit;

import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.scheduler.ExecutionContext;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmconsole.service.IndexHistoryService;
import ru.yandex.wmtools.common.util.scheduler.timetable.AbstractLockableTaskExecutor;

/**
 * Сохраняет историю показателей ТИЦ, страниц в поиске, внешних ссылок
 *
 * User: azakharov
 * Date: 26.06.12
 * Time: 13:06
 */
public class IndexHistorySaveTask extends AbstractLockableTaskExecutor {
    private static final Logger log = LoggerFactory.getLogger(IndexHistorySaveTask.class);

    private static final String INDEX_HISTORY_LOCK_NAME = "index_history_lock";

    private IndexHistoryService indexHistoryService;

    @Override
    public String runWithRELogging(ExecutionContext context) {
        if (!getLock(WMCPartition.nullPartition(), INDEX_HISTORY_LOCK_NAME, (int)TimeUnit.HOURS.toMinutes(6))) {
            String msg = "Failed to get lock for index history save task";
            log.warn(msg);
            return msg;
        }
        try {

            IndexHistoryService.IndexHistoryAliveHandler handler = new IndexHistoryService.IndexHistoryAliveHandler() {
                private DateTime lastLockRefreshTime = new DateTime();

                @Override
                public void handle() {
                    if (lastLockRefreshTime.plusHours(1).isAfterNow()) {
                        refreshLock(WMCPartition.nullPartition(), INDEX_HISTORY_LOCK_NAME);
                    }
                }
            };

            indexHistoryService.saveIndexHistory(handler);
        } catch (Exception e) {
            log.error("Exception in " + getClass().getName(), e);
        } finally {
            releaseLock(WMCPartition.nullPartition(), INDEX_HISTORY_LOCK_NAME);
        }
        return getClass().getName() + " task executed";
    }

    @Required
    public void setIndexHistoryService(IndexHistoryService service) {
        this.indexHistoryService = service;
    }
}
