package ru.yandex.wmconsole.periodic;

import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.common.scheduler.ExecutionContext;
import ru.yandex.webmaster.common.metrika.MetrikaProblemInfo;
import ru.yandex.webmaster.common.metrika.dao.TblMetrikaProblemsDao;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.service.HostInfoService;
import ru.yandex.wmconsole.service.OldWmYTService;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.util.URLUtil;
import ru.yandex.wmtools.common.util.scheduler.timetable.AbstractTaskExecutor;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;


/**
 * @author avhaliullin
 */
public class UpdateMetrikaProblemsTask extends AbstractTaskExecutor {
    private static final Logger log = LoggerFactory.getLogger(UpdateMetrikaProblemsTask.class);

    private static final int MAX_BUFFER_SIZE = 1000;
    private static final String YT_PATH = "//home/antispam/export/metrika/hosts_that_removed_ya_metrika";
    private static final DateTimeFormatter METRIKA_DATE_FORMAT = DateTimeFormat.forPattern("yyyy-MM-dd");

    private OldWmYTService ytService;
    private TblMetrikaProblemsDao tblMetrikaProblemsDao;
    private HostInfoService hostInfoService;

    @Override
    public String runWithRELogging(ExecutionContext context) throws InternalException {
        DateTime startedAt = DateTime.now();
        final List<MetrikaProblemInfo> problemsBuffer = new ArrayList<>();
        final AtomicLong processed = new AtomicLong(0L);
        ytService.readTSKV(YT_PATH, new OldWmYTService.TSKVConsumer() {
            @Override
            public void consume(Map<String, String> row) throws InternalException {
                try {
                    processed.incrementAndGet();
                    DateTime lastCheck = METRIKA_DATE_FORMAT.parseDateTime(row.get("date"));
                    long counterId = Long.parseLong(row.get("id"));
                    String hostName = URLUtil.getHostName(new URL(row.get("wm_host")), false);
                    BriefHostInfo hostInfo = hostInfoService.getBriefHostInfoByIdOrName(hostName);
                    if (hostInfo != null) {
                        problemsBuffer.add(new MetrikaProblemInfo(hostInfo.getId(), lastCheck, counterId, null));
                        if (problemsBuffer.size() >= MAX_BUFFER_SIZE) {
                            tblMetrikaProblemsDao.insertValues(problemsBuffer);
                            problemsBuffer.clear();
                        }
                    }
                } catch (MalformedURLException | InternalException e) {
                    StringBuilder sb = new StringBuilder();
                    for (Map.Entry<String, String> entry : row.entrySet()) {
                        sb.append(entry.getKey()).append("=").append(entry.getValue()).append(";");
                    }
                    throw new RuntimeException("Failed to process metrika problems. Line:" + sb.toString(), e);
                }
            }
        });
        if (!problemsBuffer.isEmpty()) {
            tblMetrikaProblemsDao.insertValues(problemsBuffer);
        }
        log.info("Processed {} lines", processed.get());
        tblMetrikaProblemsDao.deleteOldRecords(startedAt);
        return "OK";
    }

    @Required
    public void setYtService(OldWmYTService ytService) {
        this.ytService = ytService;
    }

    @Required
    public void setTblMetrikaProblemsDao(TblMetrikaProblemsDao tblMetrikaProblemsDao) {
        this.tblMetrikaProblemsDao = tblMetrikaProblemsDao;
    }

    @Required
    public void setHostInfoService(HostInfoService hostInfoService) {
        this.hostInfoService = hostInfoService;
    }
}
