package ru.yandex.webmaster.support.http.servantlet.host;

import java.net.IDN;
import java.net.URL;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.webmaster.common.WebmasterException;
import ru.yandex.webmaster.common.WebmasterExceptionType;
import ru.yandex.webmaster.common.host.HostEvent;
import ru.yandex.webmaster.common.host.HostEventService;
import ru.yandex.webmaster.common.host.HostEventType;
import ru.yandex.webmaster.common.http.Action;
import ru.yandex.webmaster.common.http.RequestFilter;
import ru.yandex.webmaster.common.http.response.HostResponse;
import ru.yandex.webmaster.common.http.response.OldPager;
import ru.yandex.webmaster.support.http.servantlet.filter.SupportUserFilter;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.service.HostInfoService;
import ru.yandex.wmconsole.util.PageUtils;
import ru.yandex.wmtools.common.SupportedProtocols;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.servantlet.AbstractServantlet;

/**
 * @author aherman
 */
public class GetHostEventsAction extends Action<GetHostEventsRequest, GetHostEventsResponse> {
    private HostEventService hostEventService;
    private HostInfoService hostInfoService;

    @Override
    public GetHostEventsResponse process(GetHostEventsRequest request) throws Exception {
        try {
            URL url = AbstractServantlet.prepareUrl(StringUtils.trim(request.getHostname()), true);
            BriefHostInfo briefHostInfo =
                    hostInfoService.getBriefHostInfoByIdOrName(SupportedProtocols.getCanonicalHostname(url));
            if (briefHostInfo == null) {
                throw new WebmasterException(WebmasterExceptionType.HOST__UNKNOWN_HOST, "Unknown host: " + request.getHostname());
            }

            Set<HostEventType> eventTypes = Collections.emptySet();
            if (request.getHostEventType() != null) {
                eventTypes = Collections.singleton(request.getHostEventType());
            }
            int eventsCount = hostEventService.countHostEvents(briefHostInfo, eventTypes, request.getEventForUserId());
            PageUtils.Pager pager = PageUtils.getPage(request.getPage(), request.getPageSize());
            List<HostEvent> events =
                    hostEventService.findHostEvents(briefHostInfo, eventTypes, request.getEventForUserId(), pager);
            OldPager oldPager = new OldPager(pager.getPage(), pager.getPageSize(), eventsCount, false);
            return new GetHostEventsResponse(eventsCount,
                    new HostResponse(briefHostInfo.getId(), briefHostInfo.getName(),
                            IDN.toUnicode(briefHostInfo.getName())), events, oldPager);
        } catch (InternalException e) {
            throw new WebmasterException(WebmasterExceptionType.INTERNAL__SQL, "Unable to get host events", e);
        }
    }

    @Override
    public List<Class<? extends RequestFilter>> getRequestFilters() {
        List<Class<? extends RequestFilter>> result = super.getRequestFilters();
        result.add(SupportUserFilter.class);
        return result;
    }

    @Required
    public void setHostEventService(HostEventService hostEventService) {
        this.hostEventService = hostEventService;
    }

    @Required
    public void setHostInfoService(HostInfoService hostInfoService) {
        this.hostInfoService = hostInfoService;
    }
}
