package ru.yandex.wmconsole.servantlet.all.about.url;

import org.springframework.beans.factory.annotation.Required;
import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.common.util.db.OrderByClause;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.all.about.url.UrlRequestId;
import ru.yandex.wmconsole.servantlet.WMCAuthenticationServantlet;
import ru.yandex.wmconsole.service.AllAboutUrlService;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.error.UserProblem;
import ru.yandex.wmtools.common.service.SupportService;
import ru.yandex.wmtools.common.util.URLUtil;

import java.net.URL;

/**
 * User: azakharov
 * Date: 21.01.14
 * Time: 14:16
 */
public abstract class AbstractSupportAllAboutUrlServantlet extends WMCAuthenticationServantlet {
    private static final String PARAM_REPORT_ID = "id";
    private static final String PARAM_HOST = "host";
    private static final String PARAM_URL = "url";

    private AllAboutUrlService allAboutUrlService;
    private SupportService supportService;

    @Override
    protected void doProcess(ServRequest req, ServResponse res, long userId) throws UserException, InternalException {
        // Пользователь является сотрудником поддержки
        if (!supportService.isSupportAndNotBlockedUser(userId)) {
            throw new UserException(UserProblem.USER_NOT_PERMITTED, "User is not support or is blocked");
        }

        // хост есть в базе данных
        BriefHostInfo hostInfo = getHostInfoByHostOrUrl(req, userId);

        // обработка конкретного запроса из серии всё о странице
        processUrlRequest(req, res, userId, hostInfo);
    }

    /**
     * Получает информацию о хосте на основе параметра host или url.
     *
     * @param req       объект запроса
     * @param userId    идентификатор пользователя
     * @return          объект с информацией о хосте
     * @throws UserException
     * @throws InternalException
     */
    private BriefHostInfo getHostInfoByHostOrUrl(
            final ServRequest req,
            final long userId) throws UserException, InternalException {

        final String host = req.getParam(PARAM_HOST);
        if (host != null && !host.isEmpty()) {
            if (host.contains("/")) {
                URL newHost = prepareHostname(host, true);
                req.setParam(PARAM_HOST, URLUtil.getHostName(newHost, false));
            }
            return getHostInfo(req, userId, false, false);
        } else {
            final String urlParam = req.getParam(PARAM_URL);
            if (urlParam == null) {
                throw new UserException(UserProblem.HOST_NOT_FOUND_IN_DATABASE, "host or url param expected but none found");
            }
            final URL url = prepareUrl(urlParam, true);
            final String hostName = URLUtil.getHostName(url, false);
            return getHostInfoService().getBriefHostInfoByIdOrName(hostName);
        }
    }

    protected OrderByClause getOrderByClause(final ServRequest req) {
        OrderByClause orderBy = new OrderByClause(
                req, allAboutUrlService.getDefaultOrderByDbField(), false,
                allAboutUrlService.getOrderByCorrespondence());
        return orderBy;
    }

    protected UrlRequestId getRequestId(final ServRequest req) throws UserException {
        return new UrlRequestId(getRequiredLongParam(req, PARAM_REPORT_ID));
    }

    protected abstract void processUrlRequest(ServRequest req, ServResponse res, long userId, BriefHostInfo hostInfo)
            throws InternalException, UserException;

    @Required
    public void setSupportService(SupportService supportService) {
        this.supportService = supportService;
    }

    @Required
    public void setAllAboutUrlService(AllAboutUrlService allAboutUrlService) {
        this.allAboutUrlService = allAboutUrlService;
    }

    public AllAboutUrlService getAllAboutUrlService() {
        return allAboutUrlService;
    }
}
