package ru.yandex.wmconsole.servantlet.drafts;

import java.net.URL;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.common.util.StringUtils;
import ru.yandex.wmconsole.data.info.UsersHostsInfo;
import ru.yandex.wmconsole.data.info.draft.messages.DraftMessageFoundHostInfo;
import ru.yandex.wmconsole.data.wrappers.draft.message.DraftMessageCheckResultWrapper;
import ru.yandex.wmconsole.servantlet.WMCAuthenticationServantlet;
import ru.yandex.wmconsole.service.ConsistentMainMirrorService;
import ru.yandex.wmconsole.service.UsersHostsService;
import ru.yandex.wmconsole.util.WwwUtil;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;

/**
 * Предварительная обработка множества хостов с приведением к главным зеркалам и WWW!WWW,
 * а также определение пользователей-владельцев для хостов в Вебмастере
 *
 * Параметры:
 * @http-param        host      строка со списком имен хостов или URL со схемой или без, в punycode или без
 * @http-param        verbose   выводить ли подробную информацию про владельцев хостов
 *
 * Результат:
 * 1) таблица в виде XML с колонками: исходный URL, хост в Вебмастере, список пользователей
 * 2) список пользователей - владельцев
 * 3) список URL, для которых не нашлось владельцев
 *
 * Для каждого URL из списка параметров
 * 1) имя хоста приводится к ASCII-представлению
 * 2) проверяется главное зеркало для хоста
 * 3) если для главного зеркала есть владельцы - добавляем в результат
 * 4) если есть владельцы для WWW!WWW-хоста - добавляем в результат
 *
 * Пример вызова:
 * http://wmconsole.dev.yandex.net:33534/draftMessageHostsCheck?_user_id=139551309&\
 * host=www.m-senin.narod.ru/index.php&\
 * host=http%3A%2F%2F%D0%BF%D1%80%D0%B5%D0%B7%D0%B8%D0%B4%D0%B5%D0%BD%D1%82.%D1%80%D1%84
 * <check-result>
 *     <host-list> <!-- выводится при наличии параметра verbose=true -->
 *         <host>
 *             <original-url>www.m-senin.narod.ru/index.php</originaAnal-url>
 *             <host>www.m-senin.narod.ru</host>
 *             <user-list>
 *                 <user>
 *                     4975146
 *                 </user>
 *             </user-list>
 *         </host>
 *         <host>
 *             <original-url>http://президент.рф</origianl-url>
 *             <host>xn--d1abbgf6aiiy.xn--p1ai</host>
 *             <user-list>
 *                 <user>
 *                     139551309
 *                 </user>
 *             </user-list>
 *         </host>
 *     </host-list>
 *     <user-list>
 *         <user>
 *              4975146
 *         </user>
 *         <user>
 *               139551309
 *         </user>
 *     </user-list>
 *     <not-successful>
 *         <host>киндза#дза</host>
 *     <not-successful/>
 * </check-result>
 *
 * User: azakharov
 * Date: 15.04.13
 * Time: 12:06
 */
public class DraftMessageHostsCheckServantlet extends WMCAuthenticationServantlet {

    private static final String PARAM_HOST = "host";
    private static final String PARAM_VERBOSE = "verbose";

    private UsersHostsService usersHostsService;
    private ConsistentMainMirrorService consistentMainMirrorService;

    @Override
    protected void doProcess(ServRequest req, ServResponse res, long userId) throws UserException, InternalException {
        final List<String> originalUrls = StringUtils.commaDelimitedListToList(req.getParam(PARAM_HOST, ""));
        final boolean verbose = getBooleanParam(req, PARAM_VERBOSE, false);

        Map<String, DraftMessageFoundHostInfo> originalToFoundHostInfo = new LinkedHashMap<String, DraftMessageFoundHostInfo>();
        List<String> notSuccessful = new LinkedList<String>();
        for (String originalUrl : originalUrls) {
            URL url = prepareUrl(originalUrl, true);
            String host = getHostName(url);

            // Проверяем, подтверждено ли у кого-нибудь главное зеркало хоста
            String mainMirror = consistentMainMirrorService.getMainMirror(host);
            Long hostId = getHostInfoService().getHostIdByHostName(mainMirror, true);
            boolean found = checkVerifiedByUsers(originalUrl, hostId, mainMirror, originalToFoundHostInfo);
            if (found) {
                continue;
            }

            // Проверяем, подтвержден ли у кого нибудь хост C/БЕЗ www
            String inversedWwwHost = WwwUtil.switchWWW(host);
            hostId = getHostInfoService().getHostIdByHostName(inversedWwwHost, true);
            found = checkVerifiedByUsers(originalUrl, hostId, inversedWwwHost, originalToFoundHostInfo);
            if (!found) {
                notSuccessful.add(originalUrl);
            }
        }

        Set<Long> userIds = new LinkedHashSet<Long>();
        for (Map.Entry<String, DraftMessageFoundHostInfo> entry: originalToFoundHostInfo.entrySet()) {
            for (UsersHostsInfo uhi : entry.getValue().getUsersHostsInfoList()) {
                userIds.add(uhi.getUserId());
            }
        }

        res.addData(new DraftMessageCheckResultWrapper(originalToFoundHostInfo, userIds, notSuccessful, verbose));
    }

    /**
     * Проверяет, подтвержден ли хост с host_id = hostId у какого нибудь пользователя.
     * Если подтвержден, то информация о подтверждении сохраняется в <code>originalToFoundHostInfo</code>
     *
     * @param originalUrl               исходный URL из запроса
     * @param hostId                    идентификатор проверяемого хоста, допускается null
     * @param hostName                  имя проверяемого хоста
     * @param originalToFoundHostInfo   MAP исходного URL в информацию о найденном хосте
     * @return                          true, если хост подтвержден у непустого множества пользователей, иначе false
     */
    private boolean checkVerifiedByUsers(String originalUrl,
                                         @Nullable Long hostId,
                                         String hostName,
                                         Map<String, DraftMessageFoundHostInfo> originalToFoundHostInfo) throws InternalException {

        if (hostId == null) {
            return false;
        }

        List<UsersHostsInfo> usersHostsList = usersHostsService.listVerifiedUsersForHost(hostId);
        if (usersHostsList.isEmpty()) {
            return false;
        }

        originalToFoundHostInfo.put(originalUrl, new DraftMessageFoundHostInfo(hostName, usersHostsList));
        return true;
    }

    @Override
    @Required
    public void setUsersHostsService(UsersHostsService usersHostsService) {
        this.usersHostsService = usersHostsService;
    }

    @Required
    public void setConsistentMainMirrorService(ConsistentMainMirrorService consistentMainMirrorService) {
        this.consistentMainMirrorService = consistentMainMirrorService;
    }
}
