package ru.yandex.wmconsole.servantlet.drafts;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.wmconsole.data.LanguageEnum;
import ru.yandex.wmconsole.data.info.DraftMessageInfo;
import ru.yandex.wmconsole.service.GlobalDraftMessagesService;
import ru.yandex.wmconsole.service.HostOwnersDraftMessagesService;
import ru.yandex.wmconsole.service.PersonalDraftMessagesService;
import ru.yandex.wmtools.common.data.wrappers.SimpleWrapper;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.servantlet.AuthenticationServantlet;
import ru.yandex.wmtools.common.util.SqlUtil;

public class DraftMessageSendServantlet extends AuthenticationServantlet {
    private static final String PARAM_MESSAGE_ID = "message";
    private static final String PARAM_HEADER = "header";
    private static final String PARAM_CONTENT = "content";
    private static final String PARAM_MODE = "mode";
    private static final String PARAM_RECIPIENTS = "recipients";
    private static final String PARAM_HOSTS = "hosts";
    private static final String PARAM_LANGUAGE = "message-lang";

    private static final String TAG_MESSAGE = "message";

    private GlobalDraftMessagesService globalDraftMessagesService;
    private PersonalDraftMessagesService personalDraftMessagesService;
    private HostOwnersDraftMessagesService hostOwnersDraftMessagesService;

    @Override
    public void doProcess(ServRequest req, ServResponse res, long userId) throws UserException, InternalException {
        long messageId = getRequiredLongParam(req, PARAM_MESSAGE_ID);
        String header = getRequiredStringParam(req, PARAM_HEADER);
        String content = getRequiredStringParam(req, PARAM_CONTENT);
        String mode = getRequiredStringParam(req, PARAM_MODE);
        String lang = req.getParam(PARAM_LANGUAGE, true);
        LanguageEnum language = LanguageEnum.fromString(lang);
        DraftMessageInfo draftMessageInfo = globalDraftMessagesService.getDraftMessage(messageId);
        // save message
        globalDraftMessagesService.saveMessage(messageId, userId, header, content, language);

        switch (draftMessageInfo.getType()) {
            case GLOBAL_MESSAGE: {
                globalDraftMessagesService.sendMessage(messageId, userId, mode);
            }
            break;
            case PERSONAL_MESSAGE: {
                String recipients = getStringParam(req, PARAM_RECIPIENTS);
                Set<String> incorrectRecipientsList = new TreeSet<String>();
                personalDraftMessagesService.addRecipients(userId, messageId, recipients, incorrectRecipientsList);

                if (incorrectRecipientsList.size() > 0) {
                    // if there are any incorrect recipients - return with error and don't send message
                    res.addData(new SimpleWrapper<String>(SqlUtil.getCommaSeparatedList(incorrectRecipientsList), "incorrect-recipients"));
                    res.addData(new SimpleWrapper<Long>(messageId, TAG_MESSAGE));
                } else {
                    // otherwise send message
                    personalDraftMessagesService.sendMessage(messageId, userId, mode);
                }
            }
            break;
            case HOST_OWNERS_MESSAGE: {
                String hosts = getStringParam(req, PARAM_HOSTS);
                List<String> incorrectHosts = new ArrayList<String>();
                if (hosts != null && !hosts.trim().isEmpty()) {
                    incorrectHosts = new ArrayList<String>();
                    List<String> hostList = Arrays.asList(hosts.split("\\s*,\\s*"));
                    hostOwnersDraftMessagesService.addHosts(userId, messageId, hostList, incorrectHosts);
                }
                if (incorrectHosts.size() > 0) {
                    res.addData(new SimpleWrapper<String>(SqlUtil.getCommaSeparatedList(incorrectHosts), "incorrect-hosts"));
                    res.addData(new SimpleWrapper<Long>(messageId, "message"));
                } else {
                    hostOwnersDraftMessagesService.sendMessage(messageId, userId, mode);
                }
            }
        }
    }

    @Required
    public void setGlobalDraftMessagesService(GlobalDraftMessagesService globalDraftMessagesService) {
        this.globalDraftMessagesService = globalDraftMessagesService;
    }

    @Required
    public void setPersonalDraftMessagesService(PersonalDraftMessagesService personalDraftMessagesService) {
        this.personalDraftMessagesService = personalDraftMessagesService;
    }

    @Required
    public void setHostOwnersDraftMessagesService(HostOwnersDraftMessagesService hostOwnersDraftMessagesService) {
        this.hostOwnersDraftMessagesService = hostOwnersDraftMessagesService;
    }
}
