package ru.yandex.wmconsole.servantlet.graphics;

import java.awt.Dimension;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.List;

import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.ChartUtilities;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.DateAxis;
import org.jfree.chart.labels.StandardXYToolTipGenerator;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.StandardXYItemRenderer;
import org.jfree.chart.renderer.xy.XYItemRenderer;
import org.jfree.data.time.Day;
import org.jfree.data.time.TimeSeries;
import org.jfree.data.time.TimeSeriesCollection;
import org.jfree.data.xy.XYDataset;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.wmconsole.data.info.DateCountInfo;
import ru.yandex.wmconsole.service.IWMCUserInfoService;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.error.UserProblem;
import ru.yandex.wmtools.common.servantlet.AuthenticationServantlet;
import ru.yandex.wmtools.common.util.TimeFilter;

/**
 * Created by IntelliJ IDEA.
 * Date: 18.02.2008
 * Time: 15:43:50
 * <p>
 * Returns JPEG of graph <quote>Number of true users</quote>,
 * i.e. users who visit our service at least once a week during last month
 *
 * <br>
 * Parameters:
 * <br>
 * reg_from optional Date (format "yyyy-MM-dd"): show from this date
 * <br>
 * reg_to optional Date (format "yyyy-MM-dd"): show till this date
 * <br>
 * show_diff optional boolean (default <code>false</code>):
 * </p>
 */
public class TrueUsersGraph2Servantlet extends AuthenticationServantlet {
    private static final String PARAM_REG_FROM = "reg_from";
    private static final String PARAM_REG_TO = "reg_to";
    private static final String PARAM_SHOW_DIFF = "show_diff";

    private IWMCUserInfoService userInfoService;
    public void setUserInfoService(final IWMCUserInfoService userInfoService) {
        this.userInfoService = userInfoService;
    }

    public byte[] createChart(boolean showDerivative, TimeFilter timeFilter) throws InternalException {
        final String chartTitle = "Число постоянных пользователей сервиса Яндекс.Вебмастер";
        final XYDataset dataset = getUsersDataset(showDerivative, timeFilter);

        final JFreeChart chart = ChartFactory.createTimeSeriesChart(chartTitle, "Дата", "Число пользователей", dataset, true, true, false);

        final XYPlot plot = chart.getXYPlot();
        final XYItemRenderer renderer = plot.getRenderer();
        renderer.setToolTipGenerator(StandardXYToolTipGenerator.getTimeSeriesInstance());
        if (renderer instanceof StandardXYItemRenderer) {
            final StandardXYItemRenderer rr = (StandardXYItemRenderer) renderer;
            rr.setShapesFilled(true);
        }

        final DateAxis axis = (DateAxis) plot.getDomainAxis();
        axis.setDateFormatOverride(new SimpleDateFormat("dd-MM-yy"));

        final ChartPanel chartPanel = new ChartPanel(chart);
        chartPanel.setPreferredSize(new Dimension(800, 570));
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        try {
            ChartUtilities.writeChartAsJPEG(outputStream, chart, 800, 600);
        }
        catch (IOException e) {
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "Cannot create writeChartAsJPEG!", e);
        }

        return outputStream.toByteArray();
    }

    private XYDataset getUsersDataset(boolean showDerivative, TimeFilter timeFilter) throws InternalException {
        TimeSeries timeSeries = new TimeSeries("Число пользователей", Day.class);
        List<DateCountInfo> date2count = userInfoService.getTrueUsersCount2(timeFilter);
        Integer prev = null;
        for (DateCountInfo info: date2count) {
            if (showDerivative) {
                int diff = info.getCount();
                if (prev != null) {
                    diff -= prev;
                    timeSeries.add(new Day(info.getDate()), diff);
                }
                prev = info.getCount();
            }
            else {
                timeSeries.add(new Day(info.getDate()), info.getCount());
            }
        }
        final TimeSeriesCollection dataset = new TimeSeriesCollection();
        dataset.addSeries(timeSeries);

        return dataset;
    }

    @Override
    public void doProcess(ServRequest req, ServResponse res, long userId) throws UserException, InternalException {
        Boolean showDerivative = req.getParamAsBoolean(PARAM_SHOW_DIFF, false);
        String fromDate = req.getParam(PARAM_REG_FROM, true);
        String toDate = req.getParam(PARAM_REG_TO, true);
        TimeFilter timeFilter;
        try {
            timeFilter = TimeFilter.create(fromDate, toDate);
        } catch (NumberFormatException e) {
            throw new UserException(UserProblem.ILLEGAL_VALUE_TYPE, "failed to parse params", e);
        }
        res.write(createChart(showDerivative, timeFilter));
    }
}
