package ru.yandex.wmconsole.servantlet.moderation.mainmirror;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.common.framework.pager.Pager;
import ru.yandex.common.util.StringUtils;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.mirror.MainMirrorHistoryInfo;
import ru.yandex.wmconsole.data.mirror.MirrorGroupActionEnum;
import ru.yandex.wmconsole.data.mirror.MirrorGroupChangeStateEnum;
import ru.yandex.wmconsole.data.wrappers.BriefHostInfoWrapper;
import ru.yandex.wmconsole.servantlet.WMCAuthenticationServantlet;
import ru.yandex.wmconsole.service.dao.TblMainMirrorHistoryDao;
import ru.yandex.wmtools.common.data.wrappers.DateWrapper;
import ru.yandex.wmtools.common.data.wrappers.UserInfoWrapper;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.error.UserProblem;
import ru.yandex.wmtools.common.util.TimeFilter;
import ru.yandex.wmtools.common.util.XmlConvertableCollectionWrapper;
import ru.yandex.wmtools.common.util.XmlDataWrapper;

import java.net.URL;
import java.text.ParseException;
import java.util.Calendar;
import java.util.List;

/**
 * User: azakharov
 * Date: 21.05.14
 * Time: 17:57
 */
public class MainMirrorHistoryServantlet extends WMCAuthenticationServantlet {

    private static final String PARAM_HOST_FILTER = "host";
    private static final String PARAM_CREATED_FROM = "created_from";
    private static final String PARAM_CREATED_TO = "created_to";
    private static final String PARAM_MODIFIED_FROM = "modified_from";
    private static final String PARAM_MODIFIED_TO = "modified_to";
    private static final String PARAM_REQUEST_STATE_FILTER = "request_state";
    private static final String PARAM_ACTION_FILTER = "action";

    private Logger log = LoggerFactory.getLogger(MainMirrorHistoryServantlet.class);

    private TblMainMirrorHistoryDao tblMainMirrorHistoryDao;

    @Override
    protected void doProcess(ServRequest req, ServResponse res, long userId) throws UserException, InternalException {
        Pager pager = createOutputStrategy(req).createPager();
        TimeFilter createdTimeFilter;
        TimeFilter modifiedTimeFilter;
        try {
            createdTimeFilter = TimeFilter.create(getStringParam(req, PARAM_CREATED_FROM), getStringParam(req, PARAM_CREATED_TO), "yyyy/MM/dd", Calendar.DAY_OF_YEAR);
            modifiedTimeFilter = TimeFilter.create(getStringParam(req, PARAM_MODIFIED_FROM), getStringParam(req, PARAM_MODIFIED_TO), "yyyy/MM/dd", Calendar.DAY_OF_YEAR);
        } catch (ParseException e) {
            throw new UserException(UserProblem.ILLEGAL_PARAM_VALUE, "Illegal value of date param", e);
        }
        String hostNameFilter = getStringParam(req, PARAM_HOST_FILTER);
        if (!StringUtils.isEmpty(hostNameFilter)) {
            try {
                final URL hostNameFilterURL = prepareHostname(hostNameFilter, true);
                hostNameFilter = getHostName(hostNameFilterURL);
            } catch (UserException ie) {
                // ignore
            }
        }

        Byte actionByte = getByteParam(req, PARAM_ACTION_FILTER);
        Byte requestStateByte = getByteParam(req, PARAM_REQUEST_STATE_FILTER);

        MirrorGroupChangeStateEnum stateFilter = null;
        if (requestStateByte != null) {
            stateFilter = MirrorGroupChangeStateEnum.R.fromValueOrNull(requestStateByte);
        }

        MirrorGroupActionEnum actionFilter = null;
        if (actionByte != null) {
            actionFilter = MirrorGroupActionEnum.R.fromValueOrNull(actionByte);
        }

        List<MainMirrorHistoryInfo> results = tblMainMirrorHistoryDao.getHistory(pager, createdTimeFilter, modifiedTimeFilter, hostNameFilter, actionFilter, stateFilter);

        // add user info from blackbox
        for (MainMirrorHistoryInfo info : results) {
            try {
                info.setUserInfo(getUserInfoService().getUserInfo(info.getUserId()));
            } catch (InternalException e) {
                log.warn("Can't get user info for userId " + info.getUserId(), e);
            } catch (UserException e) {
                log.warn("Can't get user info for userId " + info.getUserId(), e);
            }

        }

        res.addData(XmlConvertableCollectionWrapper.wrap(results, MainMirrorHistoryInfoWrapper.class, "main-mirror-change-history"));
        res.addData(pager);
    }

    public static class MainMirrorHistoryInfoWrapper extends XmlDataWrapper<MainMirrorHistoryInfo> {
        private static final String TAG_ROOT = "main-mirror-request-history";
        private static final String TAG_ACTION = "action";
        private static final String TAG_REQUEST_STATE = "request-state";
        private static final String TAG_CREATED_AT = "created-at";
        private static final String TAG_MODIFIED_AT = "modified-at";
        private static final String TAG_DESIRED_MAIN = "desired-main";
        private static final String TAG_OLD_MAIN = "old-main";

        public MainMirrorHistoryInfoWrapper(MainMirrorHistoryInfo data) {
            super(data, TAG_ROOT);
        }

        @Override
        protected void doToXml(StringBuilder result) {
            new BriefHostInfoWrapper(new BriefHostInfo(data.getHostId(), data.getHostName(), null)).toXml(result);
            putTag(result, TAG_ACTION, data.getAction().name());
            putTag(result, TAG_REQUEST_STATE, data.getState().name());
            putTag(result, TAG_DESIRED_MAIN, data.getDesiredMain());
            new DateWrapper(TAG_CREATED_AT, data.getCreatedAt()).toXml(result);
            new DateWrapper(TAG_MODIFIED_AT, data.getModifiedAt()).toXml(result);
            if (data.getUserInfo() != null) {
                new UserInfoWrapper(data.getUserInfo()).toXml(result);
            }
            putTag(result, TAG_OLD_MAIN, data.getOldMainMirrorName());
        }
    }

    @Required
    public void setTblMainMirrorHistoryDao(TblMainMirrorHistoryDao tblMainMirrorHistoryDao) {
        this.tblMainMirrorHistoryDao = tblMainMirrorHistoryDao;
    }
}
