package ru.yandex.webmaster.viewer.complaints;

import java.io.UnsupportedEncodingException;

import javax.mail.MessagingException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.io.InputStreamSource;

import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.service.MailService;

/**
 * @author aherman
 */
public class ComplaintsService {
    private static final Logger log = LoggerFactory.getLogger(ComplaintsService.class);

    private static final String PARAM_URL = "url";
    private static final String PARAM_MESSAGE = "message";
    private static final String PARAM_QUERY = "query";
    private static final String PARAM_IP = "ip";

    private String textSubjectSpam;
    private String textSubjectVirus;
    private String textSubjectUnwantedSoftware;

    private String spamToEmail;
    private String virusToEmail;

    private String defaultFromName;
    private String defaultFromEmail;

    private MailService mailService;

    public void sendComplaint(Complaint complaint, ComplaintRecipient recipient, final String attachmentFileName,
            final InputStreamSource attachmentISS)
            throws InternalException, UserException
    {
        // остальные жалобы отправляются на соответствующие рассылки по email
        String content = createContent(complaint);
        String subject = createSubject(complaint);

        ComplaintRecipient actualRecipient = getActualRecipient(complaint, recipient);

        try {
            MailService.AttachmentData[] attachments = null;

            log.debug("Send to: " + actualRecipient.getToEmail());
            log.debug("Content: " + content + "\n" +
                    "subject: " + subject + "\n" +
                    "name: " + actualRecipient.getFromName() + "\n" +
                    "email: " + actualRecipient.getFromEmail());

            if (attachmentISS != null) {
                attachments = new MailService.AttachmentData[]{new MailService.AttachmentData() {
                    @Override
                    public String getName() {
                        return attachmentFileName;
                    }

                    @Override
                    public InputStreamSource getData() {
                        return attachmentISS;
                    }
                }};
            }

            mailService.sendWithAttachments(
                    subject,
                    content,
                    new String[]{actualRecipient.getToEmail()},
                    null,
                    actualRecipient.getFromName(),
                    actualRecipient.getFromEmail(),
                    attachments
            );
        } catch (MessagingException e) {
            throw new InternalException(InternalProblem.MESSAGING_ERROR, "Messaging exception.", e);
        } catch (UnsupportedEncodingException e) {
            throw new InternalException(InternalProblem.MESSAGING_ERROR, "Unsupported encoding.", e);
        }
    }

    private ComplaintRecipient getActualRecipient(Complaint complaint, ComplaintRecipient recipient)
            throws InternalException
    {
        String toEmail = recipient.getToEmail();
        if (toEmail == null) {
            if (complaint.getType() == ComplaintType.SPAM) {
                toEmail = spamToEmail;
            } else if (complaint.getType() == ComplaintType.VIRUS) {
                toEmail = virusToEmail;
            } else if (complaint.getType() == ComplaintType.UNWANTED_SOFTWARE) {
                toEmail = virusToEmail;
            } else {
                throw new InternalException(InternalProblem.MESSAGING_ERROR, "Unknown complaint type");
            }
        }

        String fromName = recipient.getFromName();
        if (fromName == null) {
            fromName = defaultFromName;
        }

        String fromEmail = recipient.getFromEmail();
        if (fromEmail == null) {
            fromEmail = defaultFromEmail;
        }
        return new ComplaintRecipient(fromEmail, fromName, toEmail);
    }

    private String createContent(Complaint complaint) {
        StringBuilder sb = new StringBuilder();
        sb.append(PARAM_URL).append(": ").append(complaint.getUrl()).append("\n");
        sb.append(PARAM_MESSAGE).append(": ").append(complaint.getMessage()).append("\n");
        if (complaint.getQuery() != null) {

            sb.append(PARAM_QUERY).append(": ").append(complaint.getQuery()).append("\n");
        }
        if (complaint.getIp() != null) {
            sb.append(PARAM_IP).append(": ").append(complaint.getIp()).append("\n");
        }

        if (complaint.getCookies() != null) {
            sb.append("\r\nys: ").append(complaint.getCookies());
        }
        return sb.toString();
    }


    private String createSubject(Complaint complaint) {
        StringBuilder sbSubject = new StringBuilder();
        if (ComplaintType.SPAM == complaint.getType()) {
            sbSubject.append(textSubjectSpam);
        } else if (ComplaintType.VIRUS == complaint.getType()) {
            sbSubject.append(textSubjectVirus);
        } else if (ComplaintType.UNWANTED_SOFTWARE == complaint.getType()) {
            sbSubject.append(textSubjectUnwantedSoftware);
        }

        sbSubject.append(": ").append(complaint.getUrl());
        return sbSubject.toString();
    }

    @Required
    public void setMailService(MailService mailService) {
        this.mailService = mailService;
    }

    @Required
    public void setTextSubjectSpam(String textSubjectSpam) {
        this.textSubjectSpam = textSubjectSpam;
    }

    @Required
    public void setTextSubjectVirus(String textSubjectVirus) {
        this.textSubjectVirus = textSubjectVirus;
    }

    @Required
    public void setTextSubjectUnwantedSoftware(String textSubjectUnwantedSoftware) {
        this.textSubjectUnwantedSoftware = textSubjectUnwantedSoftware;
    }

    @Required
    public void setSpamToEmail(String spamToEmail) {
        this.spamToEmail = spamToEmail;
    }

    @Required
    public void setVirusToEmail(String virusToEmail) {
        this.virusToEmail = virusToEmail;
    }

    @Required
    public void setDefaultFromName(String defaultFromName) {
        this.defaultFromName = defaultFromName;
    }

    @Required
    public void setDefaultFromEmail(String defaultFromEmail) {
        this.defaultFromEmail = defaultFromEmail;
    }
}
