package ru.yandex.webmaster.viewer.complaints;

import java.io.IOException;
import java.net.URL;
import java.util.Collections;
import java.util.PriorityQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.ReadableDuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.util.concurrent.CommonThreadFactory;
import ru.yandex.wmconsole.data.wrappers.PhishingComplaintWrapper;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.servantlet.AbstractServantlet;
import ru.yandex.wmtools.common.util.HttpConnector;
import ru.yandex.wmtools.common.util.HttpResponse;

/**
 * @author aherman
 */
public class PhishingComplaintsService {
    private static final Logger log = LoggerFactory.getLogger(PhishingComplaintsService.class);

    private String phishingUrl;
    private URL phishingComplaintReceiverUrl;

    private final Lock retryQueueLock = new ReentrantLock();
    private ExecutorService resendExecutor;
    private final LinkedBlockingQueue<PhishingComplaintTask> complaintTaskQueue = new LinkedBlockingQueue<PhishingComplaintTask>();
    private final PriorityQueue<PhishingComplaintTask> retryTaskQueue = new PriorityQueue<PhishingComplaintTask>();

    private long retryThreadSleepMillis = TimeUnit.MINUTES.toMillis(5);
    private ReadableDuration nextRetryDuration;
    private int maxRetryCount;
    private int maxRetryQueueSize;
    private int maxTaskQueuSize;


    public void init() throws UserException {
        CommonThreadFactory threadFactory = new CommonThreadFactory(true, PhishingComplaintsService.class.getSimpleName() + "-");
        resendExecutor = Executors.newFixedThreadPool(2, threadFactory);
        resendExecutor.submit(new PhishingComplaintWorker());
        resendExecutor.submit(new RetryComplaintWorker());

        phishingComplaintReceiverUrl = AbstractServantlet.prepareUrl(phishingUrl, true);
    }

    public void destroy() {
        resendExecutor.shutdownNow();
    }

    /**
     * Отправляет жалобу на фишинг в спам-базу
     *
     * @param url       урл, на который жалуется
     * @param message   сообщение пользователя
     * @throws ru.yandex.wmtools.common.error.InternalException
     * @throws ru.yandex.wmtools.common.error.UserException
     */
    public void sendPhishingComplaint(String url, String message) throws InternalException,
            UserException
    {
        int queueSize = complaintTaskQueue.size();
        if (queueSize > maxTaskQueuSize) {
            throw new InternalException(InternalProblem.INTERNAL_PROBLEM,
                    "Too many phishing complaint in main queue. Discard: " + url);
        } else {
            complaintTaskQueue.add(new PhishingComplaintTask(url, message));
        }
    }

    private void sendComplaint(PhishingComplaintTask phishingComplaintTask) throws IOException {
        PhishingComplaintWrapper wrapper = new PhishingComplaintWrapper(
                Collections.singletonList(phishingComplaintTask.getUrl()),
                phishingComplaintTask.getMessage()
        );
        StringBuilder sb = new StringBuilder();
        wrapper.toXml(sb);

        HttpResponse httpResponse = new HttpConnector.RequestBuilder(phishingComplaintReceiverUrl)
                .method(HttpConnector.HttpMethod.POST)
                .entity(sb.toString(), "UTF-8")
                .okStatusRequired(true)
                .connectionTimeout(15000)
                .socketTimeout(15000)
                .execute();
        log.debug("code = " + httpResponse.getStatusCode());
        log.debug("Complaint sent: {}", phishingComplaintTask.getUrl());
    }

    private class PhishingComplaintWorker implements Runnable {
        @Override
        public void run() {
            while (!Thread.interrupted()) {
                PhishingComplaintTask phishingComplaintTask;
                try {
                    phishingComplaintTask = complaintTaskQueue.poll(5, TimeUnit.MINUTES);
                } catch (InterruptedException e) {
                    log.warn("PhishingComplaintWorker was interrupted. Stop thread.", e);
                    return;
                }

                if (phishingComplaintTask != null) {
                    log.info("Send phishing complaint: {}", phishingComplaintTask.getUrl());
                    try {
                        sendComplaint(phishingComplaintTask);
                    } catch (IOException e) {
                        log.error("Unable to send complaint: " + phishingComplaintTask.getUrl(), e);
                        postponeTask(phishingComplaintTask);
                    }
                }
            }
        }
    }

    private class RetryComplaintWorker implements Runnable {
        @Override
        public void run() {
            while (!Thread.interrupted()) {
                DateTime now = DateTime.now();
                PhishingComplaintTask phishingComplaintTask = checkDateAndPollPostponedTask(now);
                if (phishingComplaintTask != null) {
                    if (now.isAfter(phishingComplaintTask.getRetryTime())) {
                        log.info("Retry phishing complaint [{}]: {}",
                                phishingComplaintTask.getRetryCount(),
                                phishingComplaintTask.getUrl());
                        try {
                            sendComplaint(phishingComplaintTask);
                        } catch (IOException e) {
                            log.error("Unable to retry send complaint: " + phishingComplaintTask.getUrl(), e);
                            postponeTask(phishingComplaintTask);
                        }
                    }
                }

                try {
                    Thread.sleep(retryThreadSleepMillis);
                } catch (InterruptedException e) {
                    log.warn("PostponedComplaintWorker was interrupted. Stop thread.", e);
                    return;
                }
            }
        }
    }

    private PhishingComplaintTask checkDateAndPollPostponedTask(DateTime dateTime) {
        retryQueueLock.lock();
        try {
            PhishingComplaintTask task = retryTaskQueue.peek();
            if (task == null) {
                return null;
            }
            if (task.getRetryTime().isBefore(dateTime)) {
                return retryTaskQueue.poll();
            } else {
                return null;
            }
        } finally {
            retryQueueLock.unlock();
        }
    }

    private void postponeTask(PhishingComplaintTask phishingComplaintTask) {
        if (phishingComplaintTask.getRetryCount() > maxRetryCount) {
            log.error("Complaint max retry reached[{}]: {}",
                    phishingComplaintTask.getRetryCount(),
                    phishingComplaintTask.getUrl());
            return;
        }

        retryQueueLock.lock();
        try {
            if (retryTaskQueue.size() > maxRetryQueueSize) {
                log.error("Too many retry tasks, discard complaint: {}", phishingComplaintTask.getUrl());
                return;
            }

            DateTime retryTime;
            if (phishingComplaintTask.getRetryTime() == null) {
                retryTime = DateTime.now().plus(nextRetryDuration);
            } else {
                retryTime = phishingComplaintTask.getRetryTime().plus(nextRetryDuration);
            }
            phishingComplaintTask.retry(retryTime);
            log.info("Postpone complaint: next try={}, url={}", retryTime, phishingComplaintTask.getUrl());
            retryTaskQueue.add(phishingComplaintTask);
        } finally {
            retryQueueLock.unlock();
        }
    }

    private static class PhishingComplaintTask implements Comparable<PhishingComplaintTask> {
        private final String url;
        private final String message;
        private DateTime retryTime;
        private int retryCount;

        public PhishingComplaintTask(String url, String message) {
            this.url = url;
            this.message = message;
        }

        public String getUrl() {
            return url;
        }

        public String getMessage() {
            return message;
        }

        public int getRetryCount() {
            return retryCount;
        }

        private DateTime getRetryTime() {
            return retryTime;
        }

        public void retry(DateTime dateTime) {
            this.retryCount += 1;
            this.retryTime = dateTime;
        }

        @Override
        public int compareTo(PhishingComplaintTask o) {
            if (this.retryTime == null) {
                if (o.retryTime == null) {
                    return 0;
                } else {
                    return -1;
                }
            }

            if (o.retryTime == null) {
                return 1;
            }
            return this.retryTime.compareTo(o.retryTime);
        }
    }

    @Required
    public void setPhishingUrl(String phishingUrl) {
        this.phishingUrl = phishingUrl;
    }

    @Required
    public void setRetryThreadSleepMillis(long retryThreadSleepMillis) {
        this.retryThreadSleepMillis = retryThreadSleepMillis;
    }

    @Required
    public void setNextRetrySeconds(int seconds) {
        this.nextRetryDuration = Duration.standardSeconds(seconds);
    }

    @Required
    public void setMaxRetryCount(int maxRetryCount) {
        this.maxRetryCount = maxRetryCount;
    }

    @Required
    public void setMaxRetryQueueSize(int maxRetryQueueSize) {
        this.maxRetryQueueSize = maxRetryQueueSize;
    }

    @Required
    public void setMaxTaskQueuSize(int maxTaskQueuSize) {
        this.maxTaskQueuSize = maxTaskQueuSize;
    }
}
