package ru.yandex.webmaster.viewer.sitemap;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;

import ru.yandex.common.framework.pager.Pager;
import ru.yandex.common.util.db.LongRowMapper;
import ru.yandex.common.util.db.OrderByClause;
import ru.yandex.wmconsole.data.SeverityEnum;
import ru.yandex.wmconsole.data.info.HostDbHostInfo;
import ru.yandex.wmconsole.data.info.MergedSitemapInfo;
import ru.yandex.wmconsole.data.info.SitemapErrorInfo;
import ru.yandex.wmconsole.data.info.SitemapNavigationInfo;
import ru.yandex.wmconsole.data.info.sitemap.SitemapIndexInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmconsole.data.sitemaps.SitemapErrorEnum;
import ru.yandex.wmconsole.data.sitemaps.SitemapFormatEnum;
import ru.yandex.wmconsole.data.sitemaps.SitemapInfoComparator;
import ru.yandex.wmconsole.data.sitemaps.SitemapSourceEnum;
import ru.yandex.wmconsole.data.sitemaps.SitemapTypeEnum;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.servantlet.AbstractServantlet;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.util.SqlUtil;
import ru.yandex.wmtools.common.util.URLUtil;

/**
 * Сервис получения информации о последенй загруженной версии sitemap
 */
public class CurrentSitemapService extends AbstractDbService {
    private static final Logger log = LoggerFactory.getLogger(CurrentSitemapService.class);

    private static final String FIELD_SITEMAP_ID = "sitemap_id";
    private static final String FIELD_INDEX_ID = "index_id";
    private static final String FIELD_INDEX_ID_COUNT = "index_id_count";
    private static final String FIELD_INDEX_URL = "index_url";
    private static final String FIELD_URL = "url";
    private static final String FIELD_SUBMITTED_ON = "submitted_on";
    private static final String FIELD_PROCESSED_ON = "processed_on";
    private static final String FIELD_REAL_PROCESSED_ON = "real_processed_on";
    private static final String FIELD_FROM_ROBOTS_TXT = "from_robots_txt";
    private static final String FIELD_FROM_USER = "from_user";
    private static final String FIELD_FORMAT = "format";
    private static final String FIELD_URL_COUNT = "url_count";
    private static final String FIELD_URL_NAME = "url_name";
    private static final String FIELD_HOST_NAME = "host_name";
    private static final String FIELD_XML_ERROR_COUNT = "xml_error_count";
    private static final String FIELD_URL_ERROR_COUNT = "url_error_count";
    private static final String FIELD_WARNING_COUNT = "warning_count";
    private static final String FIELD_CHILDREN_COUNT = "children_count";
    public static final String FIELD_TYPE = "type";
    public static final String FIELD_SEVERITY = "severity";
    public static final String FIELD_LINE = "line";
    public static final String FIELD_POSITION = "position";
    public static final String FIELD_TEXT = "text";
    public static final String FIELD_SOURCE = "source";

    public static final String SELECT_SITEMAPS_COUNT_QUERY =
            "SELECT COUNT(*) FROM " +
                    "(SELECT s.sitemap_id FROM " +
                    "tbl_sitemaps s " +
                    "LEFT JOIN tbl_sitemaps_cur cur "+
                    "ON (s.url_name = BINARY cur.url_name AND s.host_name = cur.host_name AND s.host_id = cur.host_id) " +
                    "WHERE " +
                    "    s.host_id = ? " +
                    "AND s.from_user = 1 " +
                    "AND cur.url_name IS NULL " +
                    "UNION " +
                    "SELECT cur.sitemap_id FROM " +
                    "    tbl_sitemaps_cur cur " +
                    " LEFT JOIN (SELECT * FROM tbl_sitemaps WHERE from_user = 1 AND host_id = ?) s " + // nested select to speed up condition on different tables (s.from_user = 1 OR cur.from_robots_txt = 1)
                    " ON (s.url_name = BINARY cur.url_name AND s.host_name = cur.host_name AND s.host_id = cur.host_id) " +
                    "WHERE " +
                    "    (s.from_user = 1 OR cur.from_robots_txt = 1) " +
                    "AND cur.host_id = ? ) u";


    public static final String SELECT_SITEMAPS_QUERY =
            "SELECT " +
                    "    s.sitemap_id AS " + FIELD_SITEMAP_ID + ", " +
                    "    (SELECT si.index_id " +                            // обычно для файла sitemap 0 или 1 sitemap_index
                    "     FROM tbl_sitemaps_indexes si " +
                    "     WHERE si.sitemap_id = s.sitemap_id LIMIT 1) AS " + FIELD_INDEX_ID + ", " +
                    "    (SELECT COUNT(index_id) " +                     // если для sitemap больше 1 sitemap_index,
                    "     FROM tbl_sitemaps_indexes si " +               // то это будет обработано отдельно
                    "     WHERE si.sitemap_id = s.sitemap_id) AS " + FIELD_INDEX_ID_COUNT + ", " +
                    "    s.host_name AS host_name, " +
                    "    s.url_name AS url_name, " +
                    "    s.submitted_on AS " + FIELD_SUBMITTED_ON + ", " +
                    "    NULL AS " + FIELD_PROCESSED_ON + ", " +
                    "    NULL AS " + FIELD_REAL_PROCESSED_ON + ", " +
                    "    s.from_robots_txt AS " + FIELD_FROM_ROBOTS_TXT + ", " +
                    "    s.from_user AS " + FIELD_FROM_USER + ", " +
                    "    NULL AS " + FIELD_FORMAT + ", " +
                    "    NULL AS " + FIELD_TYPE + ", " +
                    "    NULL AS " + FIELD_URL_COUNT + ", " +
                    "    NULL AS " + FIELD_URL_ERROR_COUNT + ", " +
                    "    NULL AS " + FIELD_XML_ERROR_COUNT + ", " +
                    "    NULL AS " + FIELD_WARNING_COUNT + ", " +
                    "    NULL AS " + FIELD_CHILDREN_COUNT + ", " +
                    "    2 AS " + FIELD_SOURCE +
                    " FROM " +
                    "    tbl_sitemaps s " +
                    " LEFT JOIN "+
                    "    tbl_sitemaps_cur cur "+
                    " ON "+
                    "   (s.url_name = BINARY cur.url_name AND s.host_name = cur.host_name AND s.host_id = cur.host_id) " +
                    " WHERE " +
                    "    s.host_id = ? " +
                    " AND " +
                    "    s.from_user = 1 " +  // показываем все sitemap пользователя
                    " AND " +
                    "    cur.url_name IS NULL " +  // которых нет в последней загруженной версии
             "UNION " +
             "SELECT " +
                    "    cur.sitemap_id AS " + FIELD_SITEMAP_ID + ", " +
                    "    (SELECT index_id "+
                    "     FROM tbl_sitemaps_indexes_cur csi "+
                    "     WHERE csi.sitemap_id = cur.sitemap_id LIMIT 1) AS " + FIELD_INDEX_ID + ", " +
                    "    (SELECT COUNT(csi.index_id) " +
                    "     FROM tbl_sitemaps_indexes_cur csi "+
                    "     WHERE csi.sitemap_id = cur.sitemap_id) AS " + FIELD_INDEX_ID_COUNT + ", " +
                    "    cur.host_name AS host_name, " +
                    "    cur.url_name AS url_name, " +
                    "    cur.submitted_on AS " + FIELD_SUBMITTED_ON + ", " +
                    "    cur.processed_on AS " + FIELD_PROCESSED_ON + ", " +
                    "    cur.real_processed_on AS " + FIELD_REAL_PROCESSED_ON + ", " +
                    "    cur.from_robots_txt AS " + FIELD_FROM_ROBOTS_TXT + ", " +
                    "    COALESCE(s.from_user, 0) AS " + FIELD_FROM_USER + ", " +
                    "    cur.format AS " + FIELD_FORMAT + ", " +
                    "    cur.type AS " + FIELD_TYPE + ", " +
                    "    cur.url_count AS " + FIELD_URL_COUNT + ", " +
                    "    0 AS " + FIELD_URL_ERROR_COUNT + ", " +
                    "    (SELECT COUNT(*) FROM tbl_sitemap_cur_errors se JOIN tbl_dic_sitemap_error_type type ON se.type=type.id " +
                    " WHERE se.sitemap_id=cur.sitemap_id AND type.severity=" + SeverityEnum.ERROR.getValue() + ") AS " + FIELD_XML_ERROR_COUNT + ", " +
                    "    (SELECT COUNT(*) FROM tbl_sitemap_cur_errors se JOIN tbl_dic_sitemap_error_type type ON se.type=type.id " +
                    " WHERE se.sitemap_id=cur.sitemap_id AND type.severity=" + SeverityEnum.WARNING.getValue() + ") AS " + FIELD_WARNING_COUNT + ", " +
                    "    (SELECT COUNT(*) FROM tbl_sitemaps_indexes_cur sic WHERE sic.host_id=cur.host_id AND sic.index_id=cur.sitemap_id) AS " + FIELD_CHILDREN_COUNT + ", " +
                    "    1 AS " + FIELD_SOURCE +
                    " FROM " +
                    "    tbl_sitemaps_cur cur " +
                    " LEFT JOIN (SELECT * FROM tbl_sitemaps WHERE from_user = 1 AND host_id = ?) s " + // nested select to speed up condition on different tables (s.from_user = 1 OR cur.from_robots_txt = 1)
                    " ON (s.url_name = BINARY cur.url_name AND s.host_name = cur.host_name AND s.host_id = cur.host_id) " +
                    " WHERE " +
                    "    (s.from_user = 1 OR cur.from_robots_txt = 1) " +
                    " AND cur.host_id = ? " +
                    " %1$s ";

    private static final String SELECT_SITEMAP_INFO_QUERY =
            "SELECT " +
                    "    s.sitemap_id AS " + FIELD_SITEMAP_ID + ", " +
                    "    (SELECT csi.index_id "+
                    "     FROM tbl_sitemaps_indexes_cur csi "+
                    "     WHERE csi.sitemap_id = s.sitemap_id LIMIT 1) AS " + FIELD_INDEX_ID + ", " +
                    "    (SELECT COUNT(csi.index_id) " +
                    "     FROM tbl_sitemaps_indexes_cur csi "+
                    "     WHERE csi.sitemap_id = s.sitemap_id) AS " + FIELD_INDEX_ID_COUNT + ", " +
                    "    s.host_name AS host_name, " +
                    "    s.url_name AS url_name, " +
                    "    s.submitted_on AS " + FIELD_SUBMITTED_ON + ", " +
                    "    s.processed_on AS " + FIELD_PROCESSED_ON + ", " +
                    "    s.real_processed_on AS " + FIELD_REAL_PROCESSED_ON + ", " +
                    "    s.from_robots_txt AS " + FIELD_FROM_ROBOTS_TXT + ", " +
                    "    COALESCE(ss.from_user,0) AS " + FIELD_FROM_USER + ", " +
                    "    s.format AS " + FIELD_FORMAT + ", " +
                    "    s.type AS " + FIELD_TYPE + ", " +
                    "    s.url_count AS " + FIELD_URL_COUNT + ", " +
                    "    0 AS " + FIELD_URL_ERROR_COUNT + ", " +
                    "    (SELECT COUNT(*) FROM tbl_sitemap_cur_errors se JOIN tbl_dic_sitemap_error_type type ON se.type=type.id " +
                    "WHERE se.sitemap_id=s.sitemap_id AND type.severity=" + SeverityEnum.ERROR.getValue() + ") AS " + FIELD_XML_ERROR_COUNT + ", " +
                    "    (SELECT COUNT(*) FROM tbl_sitemap_cur_errors se JOIN tbl_dic_sitemap_error_type type ON se.type=type.id " +
                    "WHERE se.sitemap_id=s.sitemap_id AND type.severity=" + SeverityEnum.WARNING.getValue() + ") AS " + FIELD_WARNING_COUNT + ", " +
                    "    (SELECT COUNT(*) FROM tbl_sitemaps_indexes_cur sic WHERE sic.host_id=s.host_id AND sic.index_id=s.sitemap_id) AS " + FIELD_CHILDREN_COUNT + ", " +
                    "    1 AS " + FIELD_SOURCE + " " +
                    "FROM " +
                    "    tbl_sitemaps_cur s " +
                    "LEFT JOIN tbl_sitemaps ss "+
                    "ON (s.url_name = BINARY ss.url_name AND s.host_name = ss.host_name AND s.host_id = ss.host_id) " +
                    "WHERE " +
                    "    s.host_id = ? " +
                    "AND " +
                    "    s.sitemap_id = ?";

    private static final String SELECT_CHILD_SITEMAPS_COUNT_QUERY =
            "SELECT " +
                    "    COUNT(*) " +
                    "FROM " +
                    "    tbl_sitemaps_indexes_cur s " +
                    "WHERE " +
                    "    s.host_id = ? " +
                    "AND " +
                    "    s.index_id = ? "; // specified parent

    private static final String SELECT_CHILD_SITEMAPS_QUERY =
            "SELECT " +
                    "    s.sitemap_id AS " + FIELD_SITEMAP_ID + ", " +
                    "    (SELECT csi.index_id "+
                    "     FROM tbl_sitemaps_indexes_cur csi "+
                    "     WHERE csi.sitemap_id = s.sitemap_id LIMIT 1) AS " + FIELD_INDEX_ID + ", " +
                    "    (SELECT COUNT(csi.index_id) " +
                    "       FROM tbl_sitemaps_indexes_cur csi "+
                    "       WHERE csi.sitemap_id = s.sitemap_id " +
                    "    ) AS " + FIELD_INDEX_ID_COUNT + ", " +
                    "    s.host_name AS host_name, " +
                    "    s.url_name AS url_name, " +
                    "    s.submitted_on AS " + FIELD_SUBMITTED_ON + ", " +
                    "    s.processed_on AS " + FIELD_PROCESSED_ON + ", " +
                    "    s.real_processed_on AS " + FIELD_REAL_PROCESSED_ON + ", " +
                    "    s.from_robots_txt AS " + FIELD_FROM_ROBOTS_TXT + ", " +
                    "    COALESCE(ss.from_user, 0) AS " + FIELD_FROM_USER + ", " +
                    "    s.format AS " + FIELD_FORMAT + ", " +
                    "    s.type AS " + FIELD_TYPE + ", " +
                    "    s.url_count AS " + FIELD_URL_COUNT + ", " +
                    "    0 AS " + FIELD_URL_ERROR_COUNT + ", " +
                    "    (SELECT COUNT(*) "+
                    "       FROM tbl_sitemap_cur_errors se " +
                    "       JOIN tbl_dic_sitemap_error_type type " +
                    "       ON se.type=type.id " +
                    "       WHERE "+
                    "           se.sitemap_id = s.sitemap_id " +
                    "       AND "+
                    "           type.severity = " + SeverityEnum.ERROR.getValue() +
                    "    ) AS " + FIELD_XML_ERROR_COUNT + ", " +
                    "    (SELECT COUNT(*) "+
                    "       FROM tbl_sitemap_cur_errors se " +
                    "       JOIN tbl_dic_sitemap_error_type type " +
                    "       ON se.type=type.id " +
                    "       WHERE "+
                    "           se.sitemap_id = s.sitemap_id " +
                    "       AND "+
                    "           type.severity = " + SeverityEnum.WARNING.getValue() +
                    "       ) AS " + FIELD_WARNING_COUNT + ", " +
                    "    (SELECT COUNT(*) "+
                    "       FROM tbl_sitemaps_indexes_cur csi "+
                    "       WHERE "+
                    "           csi.host_id = s.host_id "+
                    "       AND "+
                    "           csi.index_id = s.sitemap_id" +
                    "    ) AS " + FIELD_CHILDREN_COUNT + ", " +
                    "    1 AS " + FIELD_SOURCE + " " +
                    "FROM " +
                    "    tbl_sitemaps_cur s " +
                    "JOIN tbl_sitemaps_indexes_cur csi " +
                    "ON (s.host_id = csi.host_id AND s.sitemap_id = csi.sitemap_id) " +
                    "LEFT JOIN tbl_sitemaps ss " +
                    "ON "+
                    "    (s.url_name = BINARY ss.url_name AND s.host_name = ss.host_name AND s.host_id = ss.host_id) " +
                    "WHERE " +
                    "    s.host_id = ? " +
                    "AND " +
                    "    csi.index_id = ? " + // specified parent
                    " %1$s ";

    private static final String SELECT_SITEMAP_ID_BY_SITEMAP_ID_IN_SEARCH_QUERY =
            "SELECT cur.sitemap_id FROM tbl_sitemaps_cur cur RIGHT JOIN tbl_sitemaps s " +
                    "ON (cur.url_name = BINARY s.url_name AND cur.host_name = s.host_name AND cur.host_id = s.host_id) " +
                    "WHERE s.sitemap_id = ? LIMIT 1";

    private static final String SELECT_SITEMAP_ID_BY_SITEMAP_ID_FROM_LATEST_QUERY =
            "SELECT s.sitemap_id FROM tbl_sitemaps_cur cur LEFT JOIN tbl_sitemaps s " +
                    "ON (cur.url_name = BINARY s.url_name AND cur.host_name = s.host_name AND cur.host_id = s.host_id) " +
                    "WHERE cur.sitemap_id = ? LIMIT 1";

    private static final String SELECT_SITEMAP_ERRORS_QUERY =
            "SELECT " +
                    "    errors.type AS " + FIELD_TYPE + ", " +
                    "    types.severity AS " + FIELD_SEVERITY + ", " +
                    "    parseErrors.line AS " + FIELD_LINE + ", " +
                    "    parseErrors.position AS " + FIELD_POSITION + ", " +
                    "    parseErrors.text AS " + FIELD_TEXT + " " +
                    "FROM " +
                    "    tbl_sitemap_cur_errors errors " +
                    "LEFT JOIN " +
                    "    tbl_dic_sitemap_error_type types " +
                    "ON " +
                    "    errors.type=types.id " +
                    "LEFT JOIN " +
                    "    tbl_sitemap_cur_parse_errors parseErrors " +
                    "ON " +
                    "    errors.error_id=parseErrors.error_id " +
                    "WHERE " +
                    "    errors.sitemap_id=? " +
                    " %1$s, errors.type, parseErrors.line, parseErrors.position " + // double "order by" clause
                    " %2$s ";

    private static final String SELECT_SITEMAP_ERRORS_COUNT_QUERY =
            "SELECT " +
                    "    COUNT(*) " +
                    "FROM " +
                    "    tbl_sitemap_cur_errors " +
                    "WHERE " +
                    "    sitemap_id=? ";

    private static final String SELECT_SITEMAP_NAVIGATION_QUERY =
            "SELECT " +
                    "    csi.index_id AS " + FIELD_INDEX_ID +", " +
                    "    CONCAT(i.host_name, i.url_name) AS " + FIELD_INDEX_URL + ", " +
                    "    s.sitemap_id AS "+ FIELD_SITEMAP_ID + ", " +
                    "    CONCAT(s.host_name, s.url_name) AS " + FIELD_URL + " " +
                    "FROM " +
                    "    tbl_sitemaps_cur s " +
                    "LEFT JOIN " +
                    "    tbl_sitemaps_indexes_cur csi " +
                    "ON " +
                    "    csi.sitemap_id = s.sitemap_id " +
                    "LEFT JOIN "+
                    "   tbl_sitemaps_cur i " +
                    "ON "+
                    "   csi.index_id = i.sitemap_id " +
                    "WHERE " +
                    "    s.sitemap_id = ? " +
                    "ORDER BY i.host_name ASC, i.url_name ASC";

    private static final String SELECT_INDEXES_QUERY =
            "SELECT " +
                    "   csi.index_id AS " + FIELD_INDEX_ID + ", " +
                    "   cs.url_name AS " + FIELD_URL_NAME + ", " +
                    "   cs.host_name AS " + FIELD_HOST_NAME + " " +
                    "FROM " +
                    "   tbl_sitemaps_indexes_cur csi " +
                    "JOIN " +
                    "   tbl_sitemaps_cur cs " +
                    "ON " +
                    "   csi.index_id = cs.sitemap_id " +
                    "WHERE " +
                    "   csi.sitemap_id = ? " +
                    "ORDER BY cs.host_name ASC, cs.url_name ASC";

    /**
     * Возвращает список sitemap'ов, включающий 1) файлы sitemap из tbl_sitemaps_cur 2) новые необработанные файлы
     * sitemap из tbl_sitemaps, для которых нет соответствующего url в tbl_sitemaps_cur
     *
     * @param hostDbHostInfo
     * @param pager
     * @param order
     * @return
     * @throws InternalException
     */
    public List<MergedSitemapInfo> getSitemapUnion(final HostDbHostInfo hostDbHostInfo, Pager pager, OrderByClause order)
            throws InternalException, UserException {
        final Long hostId = hostDbHostInfo.getHostDbHostId();
        List<MergedSitemapInfo> res = getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).
                        pageableSelect(
                                SELECT_SITEMAPS_COUNT_QUERY,
                                SELECT_SITEMAPS_QUERY,
                                mergedSitemapInfoMapper,
                                pager,
                                hostId,
                                hostId,
                                hostId);
        for (MergedSitemapInfo info : res) {
            if (info.getIndexIdCount() != null && info.getIndexIdCount() > 1) {
                info.setIndexId(calculateIndexIdForSitemapId(hostDbHostInfo, info.getId(), null));
            }
        }
        Collections.sort(res, new SitemapInfoComparator(order));
        return res;
    }

    public MergedSitemapInfo getSitemapInfo(HostDbHostInfo hostDbHostInfo, long sitemapId, Long indexId) throws InternalException, UserException {
        final List<MergedSitemapInfo> res = getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).query(
                SELECT_SITEMAP_INFO_QUERY, mergedSitemapInfoMapper, hostDbHostInfo.getHostDbHostId(), sitemapId);
        for (MergedSitemapInfo info : res) {
            if (info.getIndexIdCount() != null && info.getIndexIdCount() > 1) {
                info.setIndexId(calculateIndexIdForSitemapId(hostDbHostInfo, info.getId(), indexId));
            }
        }
        return res.isEmpty() ? null : res.get(0);
    }

    public List<MergedSitemapInfo> getSitemapListChildren(final HostDbHostInfo hostDbHostInfo, Pager pager,
                                                    OrderByClause order, long indexId) throws InternalException, UserException {
        List<MergedSitemapInfo> res = getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).
                pageableSelect(
                        SELECT_CHILD_SITEMAPS_COUNT_QUERY,
                        SELECT_CHILD_SITEMAPS_QUERY,
                        mergedSitemapInfoMapper,
                        pager,
                        hostDbHostInfo.getHostDbHostId(),
                        indexId);
        for (MergedSitemapInfo info : res) {
            if (info.getIndexIdCount() != null && info.getIndexIdCount() > 1) {
                info.setIndexId(calculateIndexIdForSitemapId(hostDbHostInfo, info.getId(), indexId));
            }
        }
        Collections.sort(res, new SitemapInfoComparator(order));

        return res;
    }

    public Long getLatestSitemapIdBySitemapIdInSearch(final HostDbHostInfo hostDbHostInfo, final long sitemapId) throws InternalException {
        final List<Long> res = getJdbcTemplate(new WMCPartition(hostDbHostInfo, null))
                .query(SELECT_SITEMAP_ID_BY_SITEMAP_ID_IN_SEARCH_QUERY, new LongRowMapper(), sitemapId);
        if (res.isEmpty()) {
            return null;
        } else {
            return res.get(0);
        }
    }

    public Long getSitemapIdInSearchByLatestSitemapId(final HostDbHostInfo hostDbHostInfo, final long sitemapId) throws InternalException {
        final List<Long> res = getJdbcTemplate(new WMCPartition(hostDbHostInfo, null))
                .query(SELECT_SITEMAP_ID_BY_SITEMAP_ID_FROM_LATEST_QUERY, new LongRowMapper(), sitemapId);
        if (res.isEmpty()) {
            return null;
        } else {
            return res.get(0);
        }
    }

    public List<SitemapErrorInfo> getSitemapErrors(Pager pager, OrderByClause order, HostDbHostInfo hostDbHostInfo, long sitemapId) throws InternalException {
        return getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).select(
                SELECT_SITEMAP_ERRORS_COUNT_QUERY, SELECT_SITEMAP_ERRORS_QUERY,
                sitemapErrorInfoRowMapper, order, pager, sitemapId);
    }

    public SitemapNavigationInfo getNavigationInfo(final HostDbHostInfo hostDbHostInfo, long sitemapId, final Long indexId) throws InternalException, UserException {
        List<SitemapNavigationInfo> res = getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).query(
                SELECT_SITEMAP_NAVIGATION_QUERY,
                sitemapNavigationInfoMapper, sitemapId);

        if (!res.isEmpty()) {
            // вначале попытаемся найти indexId, переданный через параметры (если есть)
            if (indexId != null) {
                for (SitemapNavigationInfo info : res) {
                    if (indexId.equals(info.getSitemapIndex())) {
                        return info;
                    }
                }
            }

            // затем попытаемся найти sitemapIndex для того же хоста
            for (SitemapNavigationInfo info : res) {
                // Если для sitemap отсутствует sitemapindex, то результат не пуст, а index_id = null и index_url = null
                // Это нужно, чтобы для статуса файла sitemap/sitemapindex показать имя файла
                if (info.getIndexUrl() != null) {
                    String hostName = URLUtil.getHostName(
                            AbstractServantlet.prepareUrl(info.getIndexUrl(), true), false);
                    if (hostName.equals(hostDbHostInfo.getName())) {
                        return info;
                    }
                }
            }
            return res.iterator().next();
        }

        return null;
    }

    private Long calculateIndexIdForSitemapId(HostDbHostInfo hostDbHostInfo, long sitemapId, Long indexId) throws InternalException, UserException {
        List<SitemapIndexInfo> indexList = getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).query(
                SELECT_INDEXES_QUERY, sitemapIndexMapper, sitemapId);

        if (!indexList.isEmpty()) {
            // вначале попытаемся найти indexId, переданный через параметры (если есть)
            if (indexId != null) {
                for (SitemapIndexInfo indexInfo : indexList) {
                    if (indexId.equals(indexInfo.getIndexId())) {
                        return indexId;
                    }
                }
            }

            // затем попытаемся найти sitemapIndex для того же хоста
            for (SitemapIndexInfo info : indexList) {
                String hostName = URLUtil.getHostName(
                        AbstractServantlet.prepareUrl(info.getHostName(), true), false);
                if (hostName.equals(hostDbHostInfo.getName())) {
                    return info.getIndexId();
                }
            }
            return indexList.iterator().next().getIndexId();
        }

        return null;
    }

    private static final ParameterizedRowMapper<SitemapIndexInfo> sitemapIndexMapper = new ParameterizedRowMapper<SitemapIndexInfo>() {
        @Override
        public SitemapIndexInfo mapRow(ResultSet rs, int rowNum) throws SQLException {
            return new SitemapIndexInfo(
                    rs.getLong(FIELD_INDEX_ID),
                    rs.getString(FIELD_URL_NAME),
                    rs.getString(FIELD_HOST_NAME));
        }
    };

    private static final ParameterizedRowMapper<MergedSitemapInfo> mergedSitemapInfoMapper = new ParameterizedRowMapper<MergedSitemapInfo>() {
        @Override
        public MergedSitemapInfo mapRow(ResultSet rs, int i) throws SQLException {
            Date submittedOn = SqlUtil.safeGetTimestamp(rs, FIELD_SUBMITTED_ON);
            if (rs.wasNull() || (submittedOn != null && submittedOn.getTime() <= 0)) {
                submittedOn = null;
            }

            Date processedOn = SqlUtil.safeGetTimestamp(rs, FIELD_PROCESSED_ON);
            if (rs.wasNull()) {
                processedOn = null;
            }

            Date realProcessedOn = SqlUtil.safeGetTimestamp(rs, FIELD_REAL_PROCESSED_ON);
            if (rs.wasNull()) {
                realProcessedOn = null;
            }

            SitemapFormatEnum format = SitemapFormatEnum.getByValue(rs.getInt(FIELD_FORMAT));
            if (rs.wasNull()) {
                format = null;
            }

            SitemapTypeEnum type = SitemapTypeEnum.getByValue(rs.getInt(FIELD_TYPE));
            if (rs.wasNull()) {
                type = null;
            }

            Integer xmlErrorsCount = (processedOn == null) ? null : rs.getInt(FIELD_XML_ERROR_COUNT);

            if (type == null && xmlErrorsCount != null && xmlErrorsCount > 0) {
                type = SitemapTypeEnum.SITEMAP;
            }

            Integer urlsCount = SqlUtil.getIntNullable(rs, FIELD_URL_COUNT);

            Long indexId = SqlUtil.getLongNullable(rs, FIELD_INDEX_ID);
            Long indexIdCount = rs.getLong(FIELD_INDEX_ID_COUNT);

            Integer urlErrorsCount = (processedOn == null) ? null : rs.getInt(FIELD_URL_ERROR_COUNT);
            Integer warningsCount = (processedOn == null) ? null : rs.getInt(FIELD_WARNING_COUNT);
            int childrenCount = rs.getInt(FIELD_CHILDREN_COUNT);

            String sitemapHost = rs.getString("host_name");
            String sitemapPath = rs.getString("url_name");

            final int src = rs.getInt(FIELD_SOURCE);
            final SitemapSourceEnum source;
            if (rs.wasNull()) {
                source = null;
            } else {
                source = SitemapSourceEnum.R.fromValueOrNull(src);
            }

            return new MergedSitemapInfo(
                    rs.getLong(FIELD_SITEMAP_ID),
                    indexId,
                    indexIdCount,
                    sitemapHost,
                    sitemapPath,
                    submittedOn,
                    rs.getBoolean(FIELD_FROM_ROBOTS_TXT),
                    rs.getBoolean(FIELD_FROM_USER),
                    processedOn,
                    realProcessedOn,
                    format,
                    type,
                    urlsCount,
                    xmlErrorsCount,
                    urlErrorsCount,
                    warningsCount,
                    childrenCount > 0,
                    source
            );
        }
    };

    private static final ParameterizedRowMapper<SitemapNavigationInfo> sitemapNavigationInfoMapper = new ParameterizedRowMapper<SitemapNavigationInfo>() {
        @Override
        public SitemapNavigationInfo mapRow(ResultSet rs, int rowNum) throws SQLException {
            Long indexId = rs.getLong(FIELD_INDEX_ID);
            if (rs.wasNull()) {
                indexId = null;
            }
            String indexUrl = rs.getString(FIELD_INDEX_URL);
            Long sitemapId = rs.getLong(FIELD_SITEMAP_ID);
            if (rs.wasNull()) {
                sitemapId = null;
            }
            String sitemapUrl = rs.getString(FIELD_URL);
            Integer source = SitemapSourceEnum.LATEST.getId();

            return new SitemapNavigationInfo(indexId, indexUrl, sitemapId, sitemapUrl, source);
        }
    };

    private static final ParameterizedRowMapper<SitemapErrorInfo> sitemapErrorInfoRowMapper =
            new ParameterizedRowMapper<SitemapErrorInfo>() {
                @Override
                public SitemapErrorInfo mapRow(ResultSet resultSet, int i) throws SQLException {
                    Integer line = resultSet.getInt(FIELD_LINE);
                    if (resultSet.wasNull()) {
                        line = null;
                    }
                    Integer position = resultSet.getInt(FIELD_POSITION);
                    if (resultSet.wasNull()) {
                        position = null;
                    }
                    return new SitemapErrorInfo(
                            SitemapErrorEnum.getByValue(resultSet.getInt(FIELD_TYPE)),
                            SeverityEnum.R.fromValueOrNull(resultSet.getByte(FIELD_SEVERITY)),
                            line,
                            position,
                            resultSet.getString(FIELD_TEXT)
                    );
                }
            };
}
