package ru.yandex.wmconsole.authorization;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.wmconsole.error.ClientException;
import ru.yandex.wmconsole.error.ClientProblem;
import ru.yandex.wmconsole.viewer.api.ApiVersion;
import ru.yandex.wmtools.common.authorization.AuthorizationScheme;
import ru.yandex.wmtools.common.data.info.OAuthInfo;
import ru.yandex.wmtools.common.data.info.OAuthScope;
import ru.yandex.wmtools.common.error.InternalException;

/**
 * @author aherman
 */
public class OAuthAuthorizationScheme2 implements AuthorizationScheme {
    private static final String PARAM_USER = "user";

    private final ApiOAuthPermission permission;
    private final ApiOAuthInfoService apiOAuthInfoService;

    public OAuthAuthorizationScheme2(ApiOAuthPermission permission, ApiOAuthInfoService apiOAuthInfoService) {
        this.permission = permission;
        this.apiOAuthInfoService = apiOAuthInfoService;
    }

    @Override
    public long getUserId(ServRequest req, ServResponse res) throws ClientException, InternalException {
        OAuthInfo oAuthInfo = apiOAuthInfoService.getOAuthInfo(req);
        if (acceptUser(oAuthInfo)) {
            return oAuthInfo.getUid();
        }
        OAuthScope oAuthScope = permission.getOAuthScope();

        String apiVersionStr = req.getParam(ApiVersion.API_VERSION_PARAM, ApiVersion.V1.name());
        ApiVersion apiVersion = ApiVersion.valueOf(apiVersionStr);

        if (permission != ApiOAuthPermission.VALIDATE_TOKEN_ONLY && apiVersion == ApiVersion.V1) {
            if (oAuthInfo.getUid() != getUserId(req)) {
                throw new ClientException(ClientProblem.WRONG_USER_ID, "Token doesn't match user id");
            }
        }

        String message;
        if (oAuthScope == null) {
            message = "Insufficient permissions for this operation";
        } else {
            message = "Need rights for \"" + oAuthScope.getRights() + "\"";
        }
        throw new ClientException(ClientProblem.INVALID_TOKEN, message);
    }

    public boolean acceptUser(OAuthInfo oAuthInfo) throws ClientException, InternalException {
        OAuthScope oAuthScope = permission.getOAuthScope();

        if (oAuthScope == null) {
            return true;
        }

        return oAuthInfo.getScope().contains(oAuthScope);
    }

    private long getUserId(ServRequest req) throws ClientException {
        //todo (minor) reuse AbstractServantlet.getRequired*Param() methods?
        String stringValue = req.getParam(PARAM_USER, true);
        if (stringValue == null) {
            throw new ClientException(ClientProblem.REQUIRED_PARAM_MISSED,
                    "Required param missed: " + PARAM_USER, PARAM_USER);
        }

        long userId;
        try {
            userId = Long.decode(stringValue);
        } catch (NumberFormatException e) {
            throw new ClientException(ClientProblem.ILLEGAL_VALUE_TYPE,
                    "Invalid param: " + PARAM_USER, e, PARAM_USER, stringValue);
        }

        return userId;
    }
}
