package ru.yandex.wmconsole.servantlet;

import java.net.URL;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.webmaster.common.host.dao.TblHostsMainDao;
import ru.yandex.wmconsole.data.ServerResponseInfo;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.UsersHostsInfo;
import ru.yandex.wmconsole.data.wrappers.ServerResponseInfoWrapper;
import ru.yandex.wmconsole.service.ConsistentMainMirrorService;
import ru.yandex.wmconsole.service.HttpResponseService;
import ru.yandex.wmconsole.service.error.WMCUserProblem;
import ru.yandex.wmconsole.util.WwwUtil;
import ru.yandex.wmtools.common.SupportedProtocols;
import ru.yandex.wmtools.common.error.ExtraTagInfo;
import ru.yandex.wmtools.common.error.ExtraTagNameEnum;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.error.UserProblem;
import ru.yandex.wmtools.common.sita.UserAgentEnum;

/**
 * @author Andrey Mima (amima@yandex-team.ru)
 * @author Alexey Zakharov (azakharov@yandex-team.ru)
 */
public class HttpResponseServantlet extends WMCAuthorizedHostOperationServantlet {
    private static final Logger log = LoggerFactory.getLogger(HttpResponseServantlet.class);

    private static final String PARAM_URL = "url";
    private static final String PARAM_USERAGENT = "useragent";
    private static final String PARAM_IF_MODIFIED_SINCE_ON = "modified";
    private static final String PARAM_IF_MODIFIED_SINCE_DATE = "date";

    private HttpResponseService httpResponseService;
    private TblHostsMainDao tblHostsMainDao;
    private ConsistentMainMirrorService consistentMainMirrorService;

    @Override
    protected void doProcess(ServRequest req, ServResponse res, long userId) throws UserException, InternalException {
        String urlString = getRequiredStringParam(req, PARAM_URL);

        URL url = prepareUrl(urlString, true);
        UserAgentEnum userAgent = UserAgentEnum.R.valueOfOrNull(req.getParam(PARAM_USERAGENT));
        if (userAgent == null) {
            userAgent = UserAgentEnum.ROBOT;
        }

        boolean ifModifiedSinceOn = req.getCheckBoxParamAsBoolean(PARAM_IF_MODIFIED_SINCE_ON, Boolean.FALSE);
        Date ifModifiedSince = null;
        String ifModifiedSinceDateStr = req.getParam(PARAM_IF_MODIFIED_SINCE_DATE);
        if (ifModifiedSinceOn) {
            if (StringUtils.isEmpty(ifModifiedSinceDateStr)) {
                throw new UserException(UserProblem.REQUIRED_PARAM_MISSED, "Parameter is missing", PARAM_IF_MODIFIED_SINCE_DATE);
            } else {
                try {
                    DateFormat dateFormat = new SimpleDateFormat("dd.MM.yyyy HH:mm:ss");
                    dateFormat.setTimeZone(TimeZone.getTimeZone("GMT"));
                    ifModifiedSince = dateFormat.parse(ifModifiedSinceDateStr);
                } catch (ParseException e) {
                    throw new UserException(UserProblem.ILLEGAL_PARAM_VALUE,
                            "Unable to parse date " + ifModifiedSinceDateStr,
                            new ExtraTagInfo(ExtraTagNameEnum.PARAM, PARAM_IF_MODIFIED_SINCE_DATE),
                            new ExtraTagInfo(ExtraTagNameEnum.VALUE, ifModifiedSinceDateStr));
                }
            }
        }

        final String host = SupportedProtocols.getCanonicalHostname(url);

        if (!isHostOwnedByUser(host, userId)) {
            final String mainMirror = consistentMainMirrorService.getMainMirror(host);
            if (!isHostOwnedByUser(mainMirror, userId)) {
                final String wwwSwitchedHost = WwwUtil.switchWWW(host);
                if (wwwSwitchedHost.equalsIgnoreCase(mainMirror)) {
                    throw new UserException(WMCUserProblem.HOST_NOT_OWNED_BY_USER, "Host not owned");
                } else if (!isHostOwnedByUser(wwwSwitchedHost, userId)) {
                    throw new UserException(WMCUserProblem.HOST_NOT_OWNED_BY_USER, "Host not owned");
                }
            }
        }

        // check server response
        ServerResponseInfo info = httpResponseService.getHttpResponseWithSita(
                url, userAgent, ifModifiedSinceOn, ifModifiedSince);
        res.addData(new ServerResponseInfoWrapper(info));
    }

    public boolean isHostOwnedByUser(String hostName, long userId) throws InternalException {

        final BriefHostInfo hostInfo = tblHostsMainDao.getHostIdByHostname(hostName);
        if (hostInfo == null) {
            return false;
        }

        final UsersHostsInfo usersHostsInfo = getUsersHostsService().getUsersHostsInfo(userId, hostInfo.getId());
        if (usersHostsInfo == null ||
                usersHostsInfo.getVerificationState() == null ||
                !usersHostsInfo.getVerificationState().isVerified()) {
            return false;
        }

        return true;
    }

    @Required
    public void setHttpResponseService(HttpResponseService httpResponseService) {
        this.httpResponseService = httpResponseService;
    }

    @Required
    public void setTblHostsMainDao(TblHostsMainDao tblHostsMainDao) {
        this.tblHostsMainDao = tblHostsMainDao;
    }

    @Required
    public void setConsistentMainMirrorService(ConsistentMainMirrorService consistentMainMirrorService) {
        this.consistentMainMirrorService = consistentMainMirrorService;
    }
}
