package ru.yandex.wmconsole.servantlet.api;

import java.io.IOException;
import java.io.StringReader;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import com.codahale.metrics.MetricRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.common.util.collections.Pair;
import ru.yandex.wmconsole.authorization.ApiOAuthPermission;
import ru.yandex.wmconsole.error.ClientException;
import ru.yandex.wmconsole.error.ClientProblem;
import ru.yandex.wmconsole.servantlet.WMCAuthenticationServantlet;
import ru.yandex.wmtools.common.error.InternalException;

/**
 * @author ailyin
 */
public abstract class AbstractApiServantlet extends WMCAuthenticationServantlet {
    private static final Logger log = LoggerFactory.getLogger(AbstractApiServantlet.class);

    protected MetricRegistry metricRegistry;

    @Override
    protected final void doProcess(ServRequest req, ServResponse res, long userId)
            throws ClientException, InternalException
    {
        try {
            markMeter("total", "allCalls");
            markMeter("byName." + req.getName(), "allCalls");

            apiDoProcess(req, res, userId);

            markMeter("total", "completedCalls");
            markMeter("byName." + req.getName(), "completeCalls");
        } catch (InternalException e) {
            log.error("Strange error from api", e);
            markMeter("errors." + req.getName(), e.getProblem().name());
            throw new ClientException(ClientProblem.INTERNAL_ERROR, "Internal error", e);
        }
    }

    private void markMeter(String type, String name) {
        metricRegistry.meter(String.format("api.%s.%s", type, name)).mark();
    }

    protected abstract void apiDoProcess(ServRequest req, ServResponse res, long userId)
            throws ClientException, InternalException;

    public abstract ApiOAuthPermission getRequiredPermission();

    protected String getRequiredXmlParam(ServRequest req, String path) throws ClientException {
        for (Pair<String, String> param : req.getAllParams()) {
            if (param.first.trim().startsWith("<")) {   // suppose it's xml
                try {
                    DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
                    dbf.setValidating(false);
                    dbf.setIgnoringComments(true);
                    dbf.setXIncludeAware(false);

                    dbf.setFeature(XMLConstants.FEATURE_SECURE_PROCESSING, true);
                    dbf.setFeature("http://xml.org/sax/features/external-general-entities", false);
                    dbf.setFeature("http://xml.org/sax/features/external-parameter-entities", false);
                    dbf.setFeature("http://apache.org/xml/features/nonvalidating/load-external-dtd", false);
                    dbf.setFeature("http://apache.org/xml/features/nonvalidating/load-dtd-grammar", false);
                    dbf.setFeature("http://apache.org/xml/features/validation/schema", false);

                    DocumentBuilder builder = dbf.newDocumentBuilder();
                    Document document = builder.parse(new InputSource(new StringReader(param.first)));

                    XPath xPath = XPathFactory.newInstance().newXPath();
                    Node node = (Node) xPath.evaluate(path, document, XPathConstants.NODE);

                    if (node == null) {
                        throw new ClientException(ClientProblem.ILLEGAL_PARAM_VALUE, "Incorrect xml");
                    }
                    return node.getTextContent();
                } catch (XPathExpressionException | ParserConfigurationException | SAXException | IOException e) {
                    log.error("Trying to get " + path + " Incorrect xml", e);
                    throw new ClientException(ClientProblem.ILLEGAL_PARAM_VALUE, "Incorrect xml");
                }
            }
        }

        throw new ClientException(ClientProblem.REQUIRED_PARAM_MISSED, "No data was sent");
    }

    public static String getRequiredStringParam(final ServRequest req, final String paramName) throws ClientException {
        String stringValue = req.getParam(paramName, true);
        if (stringValue == null) {
            throw new ClientException(ClientProblem.REQUIRED_PARAM_MISSED, "Required param missed: " + paramName, paramName);
        }
        return stringValue;
    }

    public static long getRequiredLongParam(final ServRequest req, final String paramName) throws ClientException {
        String stringValue = getRequiredStringParam(req, paramName);

        try {
            return Long.decode(stringValue);
        } catch (NumberFormatException e) {
            throw new ClientException(ClientProblem.ILLEGAL_VALUE_TYPE, "Invalid param: " + paramName, e, paramName, stringValue);
        }
    }

    @Required
    public void setMetricRegistry(MetricRegistry metricRegistry) {
        this.metricRegistry = metricRegistry;
    }
}
