package ru.yandex.wmconsole.servantlet.api;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.webmaster.common.host.HostEventService;
import ru.yandex.webmaster.common.host.dao.TblHostsMainDao;
import ru.yandex.webmaster.common.host.dao.TblUsersHostsDao;
import ru.yandex.webmaster.viewer.service.NewWebmasterProxyService;
import ru.yandex.webmaster.viewer.service.webmaster3.HostVerificationInfo;
import ru.yandex.wmconsole.authorization.ApiOAuthPermission;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.error.ClientException;
import ru.yandex.wmconsole.error.ClientProblem;
import ru.yandex.wmconsole.service.AddHostService;
import ru.yandex.wmconsole.viewer.api.wrappers.ApiAddHostWrapper;
import ru.yandex.wmtools.common.SupportedProtocols;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.sita.SitaRedirectInfo;

import java.net.URL;
import java.sql.Date;

/**
 * @author ailyin
 */
public class ApiAddHostServantlet extends AbstractApiServantlet {
    private static final Logger log = LoggerFactory.getLogger(ApiAddHostServantlet.class);

    private static final int MAX_USERS_HOSTS_COUNT = 1703;

    private AddHostService addHostService;
    private NewWebmasterProxyService newWebmasterProxyService;
    private TblUsersHostsDao tblUsersHostsDao;
    private TblHostsMainDao tblHostsMainDao;
    private HostEventService hostEventService;

    @Override
    public void apiDoProcess(ServRequest req, ServResponse res, long userId)
            throws ClientException, InternalException {
        String hostname = getRequiredXmlParam(req, "/host/name");

        URL url;
        try {
            url = prepareHostname(hostname, true);
        } catch (UserException e) {
            throw new ClientException(ClientProblem.INVALID_URL, "Invalid hostname: " + hostname, e);
        }

        String userIp = getStringParam(req, "_remote_ip");

        String frontendIp = req.getRemoteIp();
        String canonicalHostName = SupportedProtocols.getCanonicalHostname(url);
        addHostService.checkVerifiedRate(userId);
        addHostService.checkMainMirror(canonicalHostName, false, userId, false, false);
        SitaRedirectInfo sitaRedirectInfo = addHostService.checkHttpCode(url, false);
        addHostService.checkRedirects(url, sitaRedirectInfo, true);
        addHostService.addHostToHostDb(canonicalHostName);

        int count = tblUsersHostsDao.countHostsAddedToUser(userId);
        if (count >= MAX_USERS_HOSTS_COUNT) {
            throw new ClientException(ClientProblem.USERS_HOSTS_COUNT_LIMIT_EXCEEDED,
                    "Users host limit exceeded (" + MAX_USERS_HOSTS_COUNT + ")");
        }

        BriefHostInfo briefHostInfo = tblHostsMainDao.getHostIdByHostname(canonicalHostName);
        if (briefHostInfo == null) {
            briefHostInfo = tblHostsMainDao.addHostInfo(canonicalHostName, null);
        }
        HostVerificationInfo hostVerificationInfo = newWebmasterProxyService.addHost(userId, briefHostInfo);
        if (hostVerificationInfo.getVerificationUin() == null) {
            throw new RuntimeException("Should never happen: host " + canonicalHostName + " added for user " + userId + " to new wmc returned with null uin");
        }
        Date date = hostVerificationInfo.getVerificationDate() == null ? null : new Date(hostVerificationInfo.getVerificationDate().getTime());
        tblUsersHostsDao.addUserHostFromNewWmc(
                userId,
                briefHostInfo.getId(),
                hostVerificationInfo.getState(),
                hostVerificationInfo.getVerificationType(),
                hostVerificationInfo.getVerificationUin(),
                date
        );
        hostEventService.hostIsAdded(briefHostInfo, userId, getYaDomainParam(req), userIp, frontendIp, null);
        res.addData(new ApiAddHostWrapper(null, briefHostInfo, userId));
    }

    @Override
    public ApiOAuthPermission getRequiredPermission() {
        return ApiOAuthPermission.COMMON_OPERATIONS;
    }

    @Required
    public void setAddHostService(AddHostService addHostService) {
        this.addHostService = addHostService;
    }

    @Required
    public void setNewWebmasterProxyService(NewWebmasterProxyService newWebmasterProxyService) {
        this.newWebmasterProxyService = newWebmasterProxyService;
    }

    @Required
    public void setTblUsersHostsDao(TblUsersHostsDao tblUsersHostsDao) {
        this.tblUsersHostsDao = tblUsersHostsDao;
    }

    @Required
    public void setTblHostsMainDao(TblHostsMainDao tblHostsMainDao) {
        this.tblHostsMainDao = tblHostsMainDao;
    }

    @Required
    public void setHostEventService(HostEventService hostEventService) {
        this.hostEventService = hostEventService;
    }
}

