package ru.yandex.wmconsole.servantlet.api;

import java.util.LinkedList;
import java.util.List;

import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.webmaster.common.util.xml.SimpleXmlBuilder;
import ru.yandex.wmconsole.authorization.ApiOAuthInfoService;
import ru.yandex.wmconsole.authorization.ApiOAuthPermission;
import ru.yandex.wmconsole.authorization.OAuthAuthorizationScheme2;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.error.ClientException;
import ru.yandex.wmconsole.error.ClientProblem;
import ru.yandex.wmconsole.viewer.api.ApiLink;
import ru.yandex.wmconsole.viewer.api.ApiUrl;
import ru.yandex.wmconsole.viewer.api.ApiVersion;
import ru.yandex.wmconsole.viewer.api.ApiXmlUtils;
import ru.yandex.wmconsole.viewer.api.servantlet.history.ApiGetCrawledUrlsCountHistoryServantlet;
import ru.yandex.wmconsole.viewer.api.servantlet.history.ApiGetExcludedUrlsCountHistoryServantlet;
import ru.yandex.wmconsole.viewer.api.servantlet.history.ApiGetIncomingLinksCountHistoryServantlet;
import ru.yandex.wmconsole.viewer.api.servantlet.history.ApiGetIndexedUrlsCountHistoryServantlet;
import ru.yandex.wmconsole.viewer.api.servantlet.history.ApiGetTicHistoryServantlet;
import ru.yandex.wmconsole.viewer.api.servantlet.originals.ApiListOriginalTextsServantlet;
import ru.yandex.wmconsole.viewer.api.servantlet.sitemap.ApiListSitemapsServantlet;
import ru.yandex.wmtools.common.authorization.AuthorizationScheme;
import ru.yandex.wmtools.common.data.info.OAuthInfo;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.util.XmlDataWrapper;

/**
 * @author ailyin
 */
public class ApiHostServantlet extends AbstractApiServantlet {
    private ApiUrl apiUrl;
    private ApiOAuthInfoService apiOAuthInfoService;

    // V1 API
    private ApiHostInfoServantlet apiHostInfoServantlet;
    private ApiVerifyHostInfoServantlet apiVerifyHostInfoServantlet;
    private ApiErrorInfoServantlet apiErrorInfoServantlet;
    private AbstractApiServantlet apiIndexUrlsServantlet;
    private AbstractApiServantlet apiLinksServantlet;
    private ApiTopQueriesServantlet apiTopQueriesServantlet;

    // V2 API
    private ApiListSitemapsServantlet apiListSitemapsServantlet;
    private ApiListOriginalTextsServantlet apiListOriginalTextsServantlet;
    private ApiGetTicHistoryServantlet apiGetTicHistoryServantlet;
    private ApiGetIncomingLinksCountHistoryServantlet apiGetIncomingLinksCountHistoryServantlet;
    private ApiGetCrawledUrlsCountHistoryServantlet apiGetCrawledUrlsCountHistoryServantlet;
    private ApiGetIndexedUrlsCountHistoryServantlet apiGetIndexedUrlsCountHistoryServantlet;
    private ApiGetExcludedUrlsCountHistoryServantlet apiGetExcludedUrlsCountHistoryServantlet;

    @Override
    public void apiDoProcess(ServRequest req, ServResponse res, long userId) throws ClientException, InternalException {
        String apiVersionStr = req.getParam(ApiVersion.API_VERSION_PARAM, ApiVersion.V1.name());

        try {
            ApiVersion apiVersion = ApiVersion.valueOf(apiVersionStr);

            BriefHostInfo hostInfo = getHostInfoAndCheckHostAdded(req, userId);
            OAuthInfo oAuthInfo = apiOAuthInfoService.getOAuthInfo(req);
            List<ApiLink> apiLinks = new LinkedList<ApiLink>();

            if (apiVersion == ApiVersion.V1) {
                checkApiV1Methods(oAuthInfo, hostInfo, apiLinks);
            } else if (apiVersion == ApiVersion.V2) {
                checkApiV2Methods(oAuthInfo, hostInfo, apiLinks);
            } else if (apiVersion == ApiVersion.V3) {
                checkApiV3Methods(oAuthInfo, hostInfo, apiLinks);
            }

            res.addData(new ApiHostResponse(apiLinks, hostInfo));
        } catch (IllegalArgumentException e) {
            throw new ClientException(ClientProblem.INTERNAL_ERROR, "Missing required parameter", e);
        }
    }

    private void checkApiV1Methods(OAuthInfo oAuthInfo, BriefHostInfo hostInfo, List<ApiLink> apiLinks)
            throws ClientException, InternalException
    {
        ApiUrl.V1.User.Host userHost = apiUrl.v1().user(oAuthInfo.getUid()).hosts(hostInfo.getId());

        if (checkAuth(oAuthInfo, apiHostInfoServantlet)) {
            apiLinks.add(userHost.stats());
        }
        if (checkAuth(oAuthInfo, apiVerifyHostInfoServantlet)) {
            apiLinks.add(userHost.verify());
        }
        if (checkAuth(oAuthInfo, apiErrorInfoServantlet)) {
            apiLinks.add(userHost.excluded());
        }
        if(checkAuth(oAuthInfo, apiIndexUrlsServantlet)) {
            apiLinks.add(userHost.indexed());
        }
        if (checkAuth(oAuthInfo, apiLinksServantlet)) {
            apiLinks.add(userHost.links());
        }
        if (checkAuth(oAuthInfo, apiTopQueriesServantlet)) {
            apiLinks.add(userHost.tops());
        }
    }

    public void checkApiV2Methods(OAuthInfo oAuthInfo, BriefHostInfo hostInfo, List<ApiLink> apiLinks)
            throws ClientException, InternalException
    {
        ApiUrl.V2.Host userHost = apiUrl.v2().host(hostInfo.getId());

        if (checkAuth(oAuthInfo, apiHostInfoServantlet)) {
            apiLinks.add(userHost.stats());
        }
        if (checkAuth(oAuthInfo, apiVerifyHostInfoServantlet)) {
            apiLinks.add(userHost.verify());
        }
        if (checkAuth(oAuthInfo, apiErrorInfoServantlet)) {
            apiLinks.add(userHost.excluded());
        }
        if(checkAuth(oAuthInfo, apiIndexUrlsServantlet)) {
            apiLinks.add(userHost.indexed());
        }
        if (checkAuth(oAuthInfo, apiLinksServantlet)) {
            apiLinks.add(userHost.links());
        }
        if (checkAuth(oAuthInfo, apiTopQueriesServantlet)) {
            apiLinks.add(userHost.tops());
        }

        if (checkAuth(oAuthInfo, apiListSitemapsServantlet)) {
            apiLinks.add(userHost.sitemaps());
        }
        if (checkAuth(oAuthInfo, apiListOriginalTextsServantlet)) {
            apiLinks.add(userHost.originalTexts());
        }

        ApiUrl.V2.Host.History userHostHistory = userHost.history();
        if (checkAuth(oAuthInfo, apiGetTicHistoryServantlet)) {
            apiLinks.add(userHostHistory.tic());
        }
        if (checkAuth(oAuthInfo, apiGetIncomingLinksCountHistoryServantlet)) {
            apiLinks.add(userHostHistory.incomingLinks());
        }
        if (checkAuth(oAuthInfo, apiGetCrawledUrlsCountHistoryServantlet)) {
            apiLinks.add(userHostHistory.crawledUris());
        }
        if (checkAuth(oAuthInfo, apiGetIndexedUrlsCountHistoryServantlet)) {
            apiLinks.add(userHostHistory.indexedUrisl());
        }
        if (checkAuth(oAuthInfo, apiGetExcludedUrlsCountHistoryServantlet)) {
            apiLinks.add(userHostHistory.excludedUrls());
        }
    }

    public void checkApiV3Methods(OAuthInfo oAuthInfo, BriefHostInfo hostInfo, List<ApiLink> apiLinks)
            throws ClientException, InternalException
    {
        ApiUrl.V3.Host userHost = apiUrl.v3().host(hostInfo.getId());

        if (checkAuth(oAuthInfo, apiHostInfoServantlet)) {
            apiLinks.add(userHost.stats());
        }
        if (checkAuth(oAuthInfo, apiVerifyHostInfoServantlet)) {
            apiLinks.add(userHost.verify());
        }
        if (checkAuth(oAuthInfo, apiErrorInfoServantlet)) {
            apiLinks.add(userHost.excluded());
        }
        if(checkAuth(oAuthInfo, apiIndexUrlsServantlet)) {
            apiLinks.add(userHost.indexed());
        }
        if (checkAuth(oAuthInfo, apiLinksServantlet)) {
            apiLinks.add(userHost.links());
        }
        if (checkAuth(oAuthInfo, apiTopQueriesServantlet)) {
            apiLinks.add(userHost.tops());
        }

        if (checkAuth(oAuthInfo, apiListSitemapsServantlet)) {
            apiLinks.add(userHost.sitemaps());
        }
        if (checkAuth(oAuthInfo, apiListOriginalTextsServantlet)) {
            apiLinks.add(userHost.originalTexts());
        }

        ApiUrl.V2.Host.History userHostHistory = userHost.history();
        if (checkAuth(oAuthInfo, apiGetTicHistoryServantlet)) {
            apiLinks.add(userHostHistory.tic());
        }
        if (checkAuth(oAuthInfo, apiGetIncomingLinksCountHistoryServantlet)) {
            apiLinks.add(userHostHistory.incomingLinks());
        }
        if (checkAuth(oAuthInfo, apiGetCrawledUrlsCountHistoryServantlet)) {
            apiLinks.add(userHostHistory.crawledUris());
        }
        if (checkAuth(oAuthInfo, apiGetIndexedUrlsCountHistoryServantlet)) {
            apiLinks.add(userHostHistory.indexedUrisl());
        }
        if (checkAuth(oAuthInfo, apiGetExcludedUrlsCountHistoryServantlet)) {
            apiLinks.add(userHostHistory.excludedUrls());
        }
    }

    private boolean checkAuth(OAuthInfo oAuthInfo, AbstractApiServantlet abstractApiServantlet)
            throws ClientException, InternalException
    {
        AuthorizationScheme authorizationScheme = abstractApiServantlet.getAuthorizationScheme();
        if (!(authorizationScheme instanceof OAuthAuthorizationScheme2)) {
            return false;
        }
        OAuthAuthorizationScheme2 oAuthAuthorizationScheme2 = (OAuthAuthorizationScheme2) authorizationScheme;
        return oAuthAuthorizationScheme2.acceptUser(oAuthInfo);
    }

    private static class ApiHostResponse extends XmlDataWrapper<List<ApiLink>> {
        private final BriefHostInfo briefHostInfo;

        public ApiHostResponse(List<ApiLink> data, BriefHostInfo briefHostInfo) {
            super(data);
            this.briefHostInfo = briefHostInfo;
        }

        @Override
        protected void doToXml(StringBuilder result) {
            SimpleXmlBuilder xml = new SimpleXmlBuilder(result);
            xml.open("host");
            xml.element("name", briefHostInfo.getName());
            for (ApiLink apiLink : data) {
                ApiXmlUtils.link(xml, apiLink);
            }
            xml.close(); // host
        }
    }

    @Override
    public ApiOAuthPermission getRequiredPermission() {
        return ApiOAuthPermission.COMMON_OPERATIONS;
    }

    @Required
    public void setApiUrl(ApiUrl apiUrl) {
        this.apiUrl = apiUrl;
    }

    @Required
    public void setApiOAuthInfoService(ApiOAuthInfoService apiOAuthInfoService) {
        this.apiOAuthInfoService = apiOAuthInfoService;
    }

    @Required
    public void setApiHostInfoServantlet(ApiHostInfoServantlet apiHostInfoServantlet) {
        this.apiHostInfoServantlet = apiHostInfoServantlet;
    }

    @Required
    public void setApiVerifyHostInfoServantlet(ApiVerifyHostInfoServantlet apiVerifyHostInfoServantlet) {
        this.apiVerifyHostInfoServantlet = apiVerifyHostInfoServantlet;
    }

    @Required
    public void setApiErrorInfoServantlet(ApiErrorInfoServantlet apiErrorInfoServantlet) {
        this.apiErrorInfoServantlet = apiErrorInfoServantlet;
    }

    @Required
    public void setApiIndexUrlsServantlet(AbstractApiServantlet apiIndexUrlsServantlet) {
        this.apiIndexUrlsServantlet = apiIndexUrlsServantlet;
    }

    @Required
    public void setApiLinksServantlet(AbstractApiServantlet apiLinksServantlet) {
        this.apiLinksServantlet = apiLinksServantlet;
    }

    @Required
    public void setApiTopQueriesServantlet(ApiTopQueriesServantlet apiTopQueriesServantlet) {
        this.apiTopQueriesServantlet = apiTopQueriesServantlet;
    }

    @Required
    public void setApiListSitemapsServantlet(ApiListSitemapsServantlet apiListSitemapsServantlet) {
        this.apiListSitemapsServantlet = apiListSitemapsServantlet;
    }

    @Required
    public void setApiListOriginalTextsServantlet(ApiListOriginalTextsServantlet apiListOriginalTextsServantlet) {
        this.apiListOriginalTextsServantlet = apiListOriginalTextsServantlet;
    }

    @Required
    public void setApiGetTicHistoryServantlet(ApiGetTicHistoryServantlet apiGetTicHistoryServantlet) {
        this.apiGetTicHistoryServantlet = apiGetTicHistoryServantlet;
    }

    @Required
    public void setApiGetIncomingLinksCountHistoryServantlet(
            ApiGetIncomingLinksCountHistoryServantlet apiGetIncomingLinksCountHistoryServantlet)
    {
        this.apiGetIncomingLinksCountHistoryServantlet = apiGetIncomingLinksCountHistoryServantlet;
    }

    @Required
    public void setApiGetCrawledUrlsCountHistoryServantlet(
            ApiGetCrawledUrlsCountHistoryServantlet apiGetCrawledUrlsCountHistoryServantlet)
    {
        this.apiGetCrawledUrlsCountHistoryServantlet = apiGetCrawledUrlsCountHistoryServantlet;
    }

    @Required
    public void setApiGetIndexedUrlsCountHistoryServantlet(
            ApiGetIndexedUrlsCountHistoryServantlet apiGetIndexedUrlsCountHistoryServantlet)
    {
        this.apiGetIndexedUrlsCountHistoryServantlet = apiGetIndexedUrlsCountHistoryServantlet;
    }

    @Required
    public void setApiGetExcludedUrlsCountHistoryServantlet(
            ApiGetExcludedUrlsCountHistoryServantlet apiGetExcludedUrlsCountHistoryServantlet)
    {
        this.apiGetExcludedUrlsCountHistoryServantlet = apiGetExcludedUrlsCountHistoryServantlet;
    }
}
