package ru.yandex.wmconsole.servantlet.compare;

import java.util.List;

import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;
import org.jdom.Document;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.common.framework.pager.Pager;
import ru.yandex.common.framework.pager.SimplePagerFactory;
import ru.yandex.common.util.collections.Pair;
import ru.yandex.wmconsole.servantlet.WMCAuthenticationServantlet;
import ru.yandex.wmconsole.service.HostRegionService;
import ru.yandex.wmconsole.service.RegionsCompareService;
import ru.yandex.wmtools.common.data.info.RegionInfo;
import ru.yandex.wmtools.common.data.wrappers.JDOMWrapper;
import ru.yandex.wmtools.common.data.wrappers.RegionInfoWrapper;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.error.UserProblem;
import ru.yandex.wmtools.common.service.RegionsTreeCacheService;
import ru.yandex.wmtools.common.util.XmlConvertableCollectionWrapper;
import ru.yandex.wmtools.common.util.XmlDataWrapper;

public class RegionsCompareServantlet extends WMCAuthenticationServantlet {
    private static final String PARAM_QUERY = "query";
    private static final String PARAM_REGION_1 = "region1";
    private static final String PARAM_REGION_2 = "region2";

    protected RegionsCompareService regionsCompareService;
    protected HostRegionService hostRegionService;
    private RegionsTreeCacheService regionsTreeCacheService;

    @Override
    protected void doProcess(ServRequest req, ServResponse res, long userId) throws UserException, InternalException {
        checkService(regionsCompareService, RegionsCompareService.class);
        checkService(hostRegionService, HostRegionService.class);

        String query = getRequiredStringParam(req, PARAM_QUERY);
        if (StringUtils.isBlank(query)) {
            throw new UserException(UserProblem.ILLEGAL_PARAM_VALUE, "Empty value of 'query' is not allowed.", PARAM_QUERY, query);
        }

        int region1 = getRequiredIntParam(req, PARAM_REGION_1);
        int region2 = getRequiredIntParam(req, PARAM_REGION_2);

        SimplePagerFactory pagerFactory = new SimplePagerFactory();
        pagerFactory.setDefaultPageSize(getPageSize());
        Pager pager = pagerFactory.createPager(req);

        final Pair<String, Document> result1 = regionsCompareService.getResultForRegion(query, region1, pager);
        final Pair<String, Document> result2 = regionsCompareService.getResultForRegion(query, region2, pager);

        int count1 = regionsCompareService.extractPagerInfo(result1.getSecond());
        int count2 = regionsCompareService.extractPagerInfo(result2.getSecond());

        Pager minPager = new Pager(pager.getPageNum(), pager.getPageSize(), false);
        minPager.setCount(Math.min(count1, count2));


        res.addData(new XmlDataWrapper<String>(result1.getFirst(), "query") {
            @Override
            protected void doToXml(StringBuilder result) {
                result.append(StringEscapeUtils.escapeXml10(data));
            }
        });

        res.addData(new JDOMWrapper(result1.getSecond(), "region", "id", Long.toString(region1)));
        res.addData(new JDOMWrapper(result2.getSecond(), "region", "id", Long.toString(region2)));
        res.addData(minPager);

        // TODO if we are already using SUGGEST, don't return key regions TREE
        List<RegionInfo> regionInfos = regionsTreeCacheService.getDefaultKeyRegionsInfo();
        res.addData(XmlConvertableCollectionWrapper.wrap(regionInfos, RegionInfoWrapper.class, "key-regions"));
    }

    protected int getPageSize() {
        return 10;
    }

    @Required
    public void setRegionsCompareService(RegionsCompareService regionsCompareService) {
        this.regionsCompareService = regionsCompareService;
    }

    @Required
    public void setHostRegionService(HostRegionService hostRegionService) {
        this.hostRegionService = hostRegionService;
    }

    @Required
    public void setRegionsTreeCacheService(RegionsTreeCacheService regionsTreeCacheService) {
        this.regionsTreeCacheService = regionsTreeCacheService;
    }
}
